set scan off
prompt ##
prompt ## Creating: CLEAN_ADDRESS_APP              - PACKAGE BODY
prompt ## - Status: VALID    Last Updated : 19-AUG-2013 09:34:44
prompt ## - From  : banner_rtivm on 11-NOV-2013 14:39:28
prompt ##
Create or Replace Package Body CLEAN_Address_APP
IS
gv_Package_Body_Version  CONSTANT VARCHAR2(30) := '3.6.0.5';
gv_Package_Body_Date     CONSTANT VARCHAR2(30) := '27-Aug-2009';
/*****************************************************************************************
 *
 *  Package Name    :   CLEAN_Address_APP
 *
 *  Description     :   CLEAN_Address for custom applications implementation procedures
 *
 *  Version         :   3.6.0.4 Jul 31, 2009
 *
 * ---------------------------------------------------------------------------------------
 *   Customer has the right to modify this package so long as original copyright remains
 * ---------------------------------------------------------------------------------------
 * History:
 * 1/5/2007           - Created
 * 10/23/2008 3.5.0.5 - Added logic to skip overridden addresses in batch mode
 * 6/18/2009  3.5.1.0 - Fixed issue with non-suite fields appending to address lines - address swapping issue
 * 7/31/09    3.6.0.4 - Always put address after any non-suite field - i.e. attention above address
 * 8/27/09    3.6.0.5 - Added Address Type default to Batch processing
 *****************************************************************************************
 * Copyright (c) 2004-2008 Runner Technologies, Inc.  All Rights Reserved.
 * www.RunnerTechnologies.com   (877)784-0003  561-395-9322
 *****************************************************************************************/
-----------------------------------------------------------------------------------------
-- GLOBAL VARIABLES
-----------------------------------------------------------------------------------------
-- - NOTE: These parameters can also be set dynamically in the Set_ procedure calls
--         in the CLEAN_Address_APP package
-----------------------------------------------------------------------------------------
--
-- Set gb_expand_street_suffix to TRUE to expand the street suffix (Ave -> Avenue) if it will
-- fit on one line.  If set to FALSE, the USPS CASS standardization rules will be used.
-- DEFAULT: FALSE
gb_Expand_Street_Suffix       BOOLEAN := FALSE;
--
-- Set gb_suite_apt_before_street to TRUE to put the suite/apartment line before the address line
-- in cases where the the suite/apartment does not fit on one line.  If set to FALSE, the suite/apt
-- will appear on the line after the street address line.
-- DEFAULT: FALSE
gb_Suite_Apt_Before_Street    BOOLEAN := FALSE;
--
-- Set gb_append_suite_to_street to TRUE to always "try" to append the suite/apartment line
-- on the same line as the the address line - only when it will fit.  If set to FALSE, the
-- suite/apt will be on a separate line if there a free line available
-- DEFAULT: TRUE
gb_Append_Suite_To_Street     BOOLEAN := TRUE;
--
-- Set gb_use_ZIP_Plus4 to TRUE to append the ZIP+4 digits after the ZIP Code in the
-- ZIP field.  If set to FALSE, only the 5-digit ZIP Code will be used
-- DEFAULT: TRUE
gb_Use_ZIP_Plus4              BOOLEAN := TRUE;
--
-- Set gb_batch_update_existing to TRUE to update the existing address record when running
-- in batch mode.  If set to FALSE, a new record will be created if anything changes in the
-- address.  This could easily double the size of the Address table if set FALSE, since most
-- addresses are not entered in the USPS standards. Please take this into consideration.
-- DEFAULT: TRUE
gb_Batch_Update_Existing      BOOLEAN := TRUE;
--
-- Set gn_telephone_distance_limit to the number of miles threshhold to allow before an
-- alert message is displayed to the user.  This is the distance between the telephone
-- wire center (area code and prefix = first 6 digits) and the 5-digit ZIP Code
-- DEFAULT: 100
gn_Telephone_Distance_Limit   NUMBER  := 100;
--
-- Set gn_address_lines to the number of address lines to be processed
-- DEFAULT: 4
gn_Address_Lines              NUMBER  := 4;
--
-- Set gn_max_address_length to the size of the address fields in Address table (currently 55)
-- DEFAULT: 55
gn_Max_Address_Length         NUMBER  := 55;
--
-- Set gn_max_city_length to the size of the city field in Address table (currently 30)
-- DEFAULT: 30
gn_Max_City_Length            NUMBER  := 30;
--
-- Set gn_max_state_length to the size of the state field in Address table (currently 30)
-- DEFAULT: 6
gn_Max_State_Length            NUMBER  := 6;
--
-- Set gn_error_suggest_count to the number of possible matches to return for unverified addresses
gn_Error_Suggest_Count        PLS_INTEGER := 0;
--
-- Set gb_Upper_Case_Results to upper case all address results
gb_Upper_Case_Results         BOOLEAN := FALSE;
--
-- Set gb_Ignore_Error to TRUE to allow Web Self-Service to save data even with an error code
-- NOTE: The error code will still be stored on the record but allow the transaction to continue
-- The functions Set_Ignore_Error and Get_Ignore_Error are used for this variable
gb_Ignore_Error               BOOLEAN := FALSE;
--
-- gv_Object_Name contains the Object name that is currently calling the verification procedure
gv_Object_Name                VARCHAR2(100) := NULL;
--
-- Set gv_Local_Country to the Application specific Country Code for the predominant country
gv_Local_Country              VARCHAR2(100) := 'USA';
--
-- Set gb_Skip_International TRUE to not process international address records for interactive processing
-- This allows more control on the purchased Web Service transactions.
-- NOTE: User should pass this parameter into the batch procedure rather than setting directly
gb_Skip_International         BOOLEAN := TRUE;
--
-------------------------------------------------------------------------------
--
-- TYPES and PROCEDURES for managing result output
--
-------------------------------------------------------------------------------
-- Declare results type table for holding summary
TYPE RESULTS_REC IS RECORD (
   code             VARCHAR2(10)
  ,description      VARCHAR2(100)
  ,rec_count        NUMBER
  );
TYPE RESULTS_TAB IS TABLE OF RESULTS_REC INDEX BY BINARY_INTEGER;
gt_results        RESULTS_TAB;
gi_total          NUMBER := 0;
-----------------------------------------------------------------------------------------
-- PRIVATE PROCEDURES
-----------------------------------------------------------------------------------------
/******************************************************************************************
 *  Procedure Name  :   Add_Result
 *
 *  Scope           :   PRIVATE
 *
 *  Description     :   Add a result code and string to internal buffer
 ******************************************************************************************/
PROCEDURE Add_Result (
                   pv_code        IN VARCHAR2
                  ,pv_description IN VARCHAR2
                  )
IS
  lb_found BOOLEAN := FALSE;
  li       BINARY_INTEGER;
BEGIN
  gi_total := gi_total + 1;
  FOR li in 1..gt_results.COUNT LOOP
    if gt_results(li).code = NVL(pv_code, '.') then
      gt_results(li).rec_count := gt_results(li).rec_count + 1;
      lb_found := TRUE;
      exit;
    end if;
  END LOOP;
  if NOT lb_found then
    li := gt_results.COUNT + 1;
    gt_results(li).code        := NVL(pv_code, '.');
    gt_results(li).description := NVL(substr(pv_description,1,100), '.');
    gt_results(li).rec_count   := 1;
  end if;
END Add_Result;
/******************************************************************************************
 *  Procedure Name  :   Delete_Results
 *
 *  Scope           :   PRIVATE
 *
 *  Description     :   Delete the results table in memory
 ******************************************************************************************/
PROCEDURE Delete_Results
IS
BEGIN
  gt_results.DELETE;
  gi_total := 0;
END Delete_Results;
/******************************************************************************************
 *  Procedure Name  :   Show_Results
 *
 *  Scope           :   PRIVATE
 *
 *  Description     :   Show the results table in DBMS_OUTPUT
 ******************************************************************************************/
PROCEDURE Show_Results
IS
  li       BINARY_INTEGER;
  li_total NUMBER := 0;
BEGIN
  dbms_output.put_line('---------------------------');
  dbms_output.put_line('--     ERROR SUMMARY     --');
  dbms_output.put_line('---------------------------');
  dbms_output.put_line(
        rpad('Code' ,11)
      ||rpad('Description', 51)
      ||'Count'
      );
  dbms_output.put_line(
        rpad('-' ,10,'-')||' '
      ||rpad('-', 50,'-')||' '
      ||'------------------'
      );
  FOR li in 1..gt_results.COUNT LOOP
    if gi_total = 0 then
      dbms_output.put_line(
          rpad(NVL(gt_results(li).code,'.'),11)
        ||rpad(substr(gt_results(li).description,1,50),51)
        ||to_char(gt_results(li).rec_count)
        );
    else
    dbms_output.put_line(
        rpad(NVL(gt_results(li).code,'.'),11)
      ||rpad(substr(gt_results(li).description,1,50),51)
      ||to_char(gt_results(li).rec_count)
        ||' ('||to_char(round(100*gt_results(li).rec_count/gi_total,1))||'%)'
      );
    end if;
    li_total := li_total + gt_results(li).rec_count;
    if length(gt_results(li).description) > 50 then
      dbms_output.put_line(
          rpad(' ',11)
        ||rpad(substr(gt_results(li).description,51,50),51)
        );
    end if;
    if length(gt_results(li).description) > 100 then
      dbms_output.put_line(
          rpad(' ',11)
        ||rpad(substr(gt_results(li).description,101,50),51)
        );
    end if;
  END LOOP;
  -- Show total
  dbms_output.put_line(
        rpad('-' ,10,'-')||' '
      ||rpad('-', 50,'-')||' '
      ||'----------'
      );
  dbms_output.put_line(
        rpad('Total:' ,61)||' '
      ||to_char(li_total)
      );
END Show_Results;
/******************************************************************************************
 *  Procedure Name  :   Trim_Address_Line
 *
 *  Scope           :   PRIVATE
 *
 *  Description     :   Do a smart trim on address lines over 55 characters instead of truncing
 *                      - trim at the last space before the max length
 ******************************************************************************************/
PROCEDURE Trim_Address_Line (
                     fv_address                     IN OUT VARCHAR2
                    ,fn_max_length                  IN     NUMBER   DEFAULT NULL
                    )
IS
  ln_space  PLS_INTEGER;
  ln_max    PLS_INTEGER;
BEGIN
  ln_max := NVL(fn_max_length, gn_Max_Address_Length);
  WHILE NVL(length(fv_address),0) > ln_max LOOP
    ln_space := instr(substr(fv_address,1,ln_max+1), ' ', -1);
    if ln_space > 1 then
      -- trim at the last space
      fv_address := substr(fv_address, 1, ln_space-1);
    else
      -- last resort - do the trunc
      fv_address := substr(fv_address, 1, ln_max);
    end if;
  END LOOP;
END Trim_Address_Line;
-----------------------------------------------------------------------------------------
-- PUBLIC PROCEDURES
-----------------------------------------------------------------------------------------
/******************************************************************************************
 *  Procedure Name  :   Verify_CLEAN_Address_Record
 *
 *  Scope           :   PUBLIC
 *
 *  Description     :   Verify a single address for Application
 *                      - accepts CLEAN_Address Address record type as input
 *                      - Assumes address is stored in generic address lines:
 *                          Address_Line_1-4, City, State, Postal_Code, Country_Code
 ******************************************************************************************/
PROCEDURE Verify_CLEAN_Address_Record (
                     f_address_rec                  IN OUT CLEAN_Address.Address_rec
                    ,fv_plsql_error                 IN OUT VARCHAR2
                    ,fv_address_type                IN     VARCHAR2 DEFAULT NULL /* ADDRESS_TYPE */
                    ,fv_object_name                 IN     VARCHAR2 DEFAULT NULL /* Object Name where procedure is called from */
                    )
IS
  --
  -- Cursor to get the Application Country Description based on country code
  --
  cursor c_nation (pv_country_code VARCHAR2) IS
    select DESCRIPTION
    from   CLN_COUNTRIES_V
    where  COUNTRY_CODE = pv_country_code;
  lv_country_code_backup VARCHAR2(100) := NULL;
  lb_verified            BOOLEAN       := FALSE;
  lb_suite_appended      BOOLEAN       := FALSE;
  lb_input_Address_Name  BOOLEAN     := FALSE;
  lv_parsed_address      VARCHAR2(100);
  lb_company_input       BOOLEAN       := FALSE;
  l_address_rec_bak      CLEAN_Address.Address_rec;
BEGIN
  fv_plsql_error := NULL;
  -- 3/6/08 - Set the object name so user exit procedure can identify where the call was made from
  Set_Object_Name(fv_object_name);
  -- Backup existing Application country code
  lv_country_code_backup := f_address_rec.country_code;
  l_address_rec_bak      := f_address_rec;
  -- 3/1/08 - Get the Nation name for international address verifications instead of the code
  --          because many institutions do not use the ISO-2, ISO-3, or ISO-Numeric format for country code
  if     f_address_rec.country_code IS NOT NULL
     -- Don't pull country name back for US and Canada since that data is accessed locally
     and f_address_rec.country_code NOT IN ('US', 'USA', 'U.S.', 'U.S.A.', '840', '157')
     and f_address_rec.country_code NOT IN ('CA', 'CAN', 'CANADA', '124')
     then
    open c_nation(f_address_rec.country_code);
    fetch c_nation
    into  f_address_rec.country_name;
    close c_nation;
  end if;
  -- Set boolean if Company is present on Input - use for verification but don't add to address lines
  if f_address_rec.company IS NOT NULL then
    lb_company_input := TRUE;
  end if;
  --
  -- Verify the address
  --
  BEGIN
    if trim(f_address_rec.Address_Name) IS NOT NULL then
      lb_input_Address_Name := TRUE;
    end if;
    --
    -- Pre Address Verify User Exit
    -- - Make any custom changes in the CLEAN_Address_APP_UE User Exit package
    --
    CLEAN_Address_APP_UE.Pre_Address_Verify(
             f_address_rec     => f_address_rec
            ,fv_plsql_error    => fv_plsql_error
            ,fv_address_type   => fv_address_type
            ,fb_verified       => lb_verified
            );
    --
    -- Exit if the address was verified or overridden in the User Exit
    --
    if lb_verified then
      return;
    end if;
    --
    -- Remove country code for verifying international addresses - use country name exclusively for better match
    --
    if lv_country_code_backup IS NOT NULL
       and f_address_rec.country_name IS NOT NULL then
      f_address_rec.country_code := NULL;
    end if;
    --
    -- Support for up to 7 address lines - use only last 4 and restore the rest from the backup record after the call
    --
    if trim(f_address_rec.Address_Line_7) IS NOT NULL then
      f_address_rec.Address_Line_1 := f_address_rec.Address_Line_4;
      f_address_rec.Address_Line_2 := f_address_rec.Address_Line_5;
      f_address_rec.Address_Line_3 := f_address_rec.Address_Line_6;
      f_address_rec.Address_Line_4 := f_address_rec.Address_Line_7;
      f_address_rec.Address_Line_5 := NULL;
      f_address_rec.Address_Line_6 := NULL;
      f_address_rec.Address_Line_7 := NULL;
    elsif trim(f_address_rec.Address_Line_6) IS NOT NULL then
      f_address_rec.Address_Line_1 := f_address_rec.Address_Line_3;
      f_address_rec.Address_Line_2 := f_address_rec.Address_Line_4;
      f_address_rec.Address_Line_3 := f_address_rec.Address_Line_5;
      f_address_rec.Address_Line_4 := f_address_rec.Address_Line_6;
      f_address_rec.Address_Line_5 := NULL;
      f_address_rec.Address_Line_6 := NULL;
    elsif trim(f_address_rec.Address_Line_5) IS NOT NULL then
      f_address_rec.Address_Line_1 := f_address_rec.Address_Line_2;
      f_address_rec.Address_Line_2 := f_address_rec.Address_Line_3;
      f_address_rec.Address_Line_3 := f_address_rec.Address_Line_4;
      f_address_rec.Address_Line_4 := f_address_rec.Address_Line_5;
      f_address_rec.Address_Line_5 := NULL;
    end if;
    -- 7/31/09 - Fix Attn being dropped from line 3
    if upper(f_address_rec.Address_Line_3) like 'ATTN%'
       and trim(f_address_rec.Address_Line_4) IS NULL
       then
      f_address_rec.Address_Line_4 := f_address_rec.Address_Line_3;
      f_address_rec.Address_Line_3 := f_address_rec.Address_Line_2;
      f_address_rec.Address_Line_2 := f_address_rec.Address_Line_1;
      f_address_rec.Address_Line_1 := f_address_rec.Address_Line_4;
      f_address_rec.Address_Line_4 := NULL;
    end if;
    -- 7/31/09 - Fix "P O Box" duplication
    if upper(f_address_rec.Address_Line_1) like 'P O BOX%' then
      f_address_rec.Address_Line_1 := 'PO Box '||substr(f_address_rec.Address_Line_1, 8);
    end if;
    if upper(f_address_rec.Address_Line_2) like 'P O BOX%' then
      f_address_rec.Address_Line_2 := 'PO Box '||substr(f_address_rec.Address_Line_2, 8);
    end if;
    if upper(f_address_rec.Address_Line_3) like 'P O BOX%' then
      f_address_rec.Address_Line_3 := 'PO Box '||substr(f_address_rec.Address_Line_3, 8);
    end if;
    -- 7/31/09 - Fix Address on first line of 3 lines issue
    if     substr(trim(f_address_rec.Address_Line_1),1,1)     between '0' and '9'
       and substr(trim(f_address_rec.Address_Line_2),1,1) NOT between '0' and '9'
       and substr(trim(f_address_rec.Address_Line_3),1,1) NOT between '0' and '9'
       and trim(f_address_rec.Address_Line_4) IS NULL
       then
      f_address_rec.Address_Line_4 := f_address_rec.Address_Line_1;
      f_address_rec.Address_Line_1 := f_address_rec.Address_Line_2;
      f_address_rec.Address_Line_2 := f_address_rec.Address_Line_3;
      f_address_rec.Address_Line_3 := f_address_rec.Address_Line_4;
      f_address_rec.Address_Line_4 := NULL;
    end if;
    --
    -- Process an Address with Standardized component fields
    --
    CLEAN_Address.Verify_Generic(
           f_address_rec              => f_address_rec
          ,fn_number_address_lines    => gn_Address_Lines
          ,fb_generic_address_name    => FALSE
          ,fb_generic_company         => FALSE
          ,fb_address_suite_same_line => FALSE
          ,fn_Error_Suggest_Count     => gn_Error_Suggest_Count
          ,fb_Verbose_Suggest_list    => FALSE
          ,fb_upper_case_results      => gb_Upper_Case_Results
          );
    -- Set the error string to NULL if there is no error
    if upper(f_address_rec.error_string) = 'NO ERROR' then
      f_address_rec.error_string := NULL;
    end if;
    --
    -- Fix the last line if present in the returned address line
    -- Don't return line 4 as the city/state/ZIP line (last_line)
    --
    if f_address_rec.address_line_4 = f_address_rec.last_line then
      f_address_rec.address_line_4 := NULL;
      -- 7/31/09 check for missing suite issue
      if f_address_rec.suite IS NOT NULL
         and f_address_rec.Address_Line_1||' '||f_address_rec.Address_Line_2||' '||f_address_rec.Address_Line_3 NOT LIKE '%'||f_address_rec.suite||'%'
         then
        f_address_rec.address_line_4 := f_address_rec.suite;
      end if;
    end if;
    -- 7/31/09 Fix duplicate Company name issue
    if NOT lb_company_input
       and ( f_address_rec.company = f_address_rec.Address_Line_1
           or f_address_rec.company = f_address_rec.Address_Line_2
           or f_address_rec.company = f_address_rec.Address_Line_3
           or f_address_rec.company = f_address_rec.Address_Line_4
           )
       then
      f_address_rec.company := NULL;
    end if;
    -- 3/1/08 - Make sure address data was returned for international verification, otherwise restore original record
    if   f_address_rec.Address_Line_1||f_address_rec.Address_Line_2||f_address_rec.Address_Line_3
       ||f_address_rec.City||f_address_rec.State||f_address_rec.Postal_code IS NULL then
      -- Restore original input address
      f_address_rec.Company        := l_address_rec_bak.company;
      f_address_rec.Address_Line_1 := l_address_rec_bak.address_line_1;
      f_address_rec.Address_Line_2 := l_address_rec_bak.address_line_2;
      f_address_rec.Address_Line_3 := l_address_rec_bak.address_line_3;
      f_address_rec.Address_Line_4 := l_address_rec_bak.address_line_4;
      f_address_rec.Address_Line_5 := l_address_rec_bak.address_line_5;
      f_address_rec.Address_Line_6 := l_address_rec_bak.address_line_6;
      f_address_rec.Address_Line_7 := l_address_rec_bak.address_line_7;
      f_address_rec.City           := l_address_rec_bak.City;
      f_address_rec.State          := l_address_rec_bak.State;
      f_address_rec.Province       := l_address_rec_bak.Province;
      f_address_rec.Postal_code    := l_address_rec_bak.postal_code;
      f_address_rec.country_code   := l_address_rec_bak.country_code;
    end if;
    -- Restore the backed up country code in case it was reset by international address verification
    if lv_country_code_backup IS NOT NULL
       and NVL(f_address_rec.country_code,lv_country_code_backup) != lv_country_code_backup
       and NVL(f_address_rec.country_code,'US') != 'US'
       then
      f_address_rec.country_code := lv_country_code_backup;
    end if;
    --
    -- Post Address Verify User Exit
    -- - Make any custom changes in the CLEAN_Address_APP_UE User Exit package
    --
    CLEAN_Address_APP_UE.Post_Address_Verify(
           f_address_rec     => f_address_rec
          ,fv_plsql_error    => fv_plsql_error
          ,fv_address_type   => fv_address_type
          ,fb_verified       => lb_verified
          );
    --
    -- Exit if the address was verified or overridden in the User Exit
    --
    if lb_verified then
      --
      -- Support for up to 7 address lines - Restore previously backed up address lines toward the top
      --
      if trim(l_address_rec_bak.Address_Line_7) IS NOT NULL then
        f_address_rec.Address_Line_7 := f_address_rec.Address_Line_4;
        f_address_rec.Address_Line_6 := f_address_rec.Address_Line_3;
        f_address_rec.Address_Line_5 := f_address_rec.Address_Line_2;
        f_address_rec.Address_Line_4 := f_address_rec.Address_Line_1;
        f_address_rec.Address_Line_1 := l_address_rec_bak.Address_Line_1;
        f_address_rec.Address_Line_2 := l_address_rec_bak.Address_Line_2;
        f_address_rec.Address_Line_3 := l_address_rec_bak.Address_Line_3;
     elsif trim(l_address_rec_bak.Address_Line_6) IS NOT NULL then
        f_address_rec.Address_Line_6 := f_address_rec.Address_Line_4;
        f_address_rec.Address_Line_5 := f_address_rec.Address_Line_3;
        f_address_rec.Address_Line_4 := f_address_rec.Address_Line_2;
        f_address_rec.Address_Line_3 := f_address_rec.Address_Line_1;
        f_address_rec.Address_Line_1 := l_address_rec_bak.Address_Line_1;
        f_address_rec.Address_Line_2 := l_address_rec_bak.Address_Line_2;
      elsif trim(l_address_rec_bak.Address_Line_5) IS NOT NULL then
        f_address_rec.Address_Line_5 := f_address_rec.Address_Line_4;
        f_address_rec.Address_Line_4 := f_address_rec.Address_Line_3;
        f_address_rec.Address_Line_3 := f_address_rec.Address_Line_2;
        f_address_rec.Address_Line_2 := f_address_rec.Address_Line_1;
        f_address_rec.Address_Line_1 := l_address_rec_bak.Address_Line_1;
      end if;
      return;
    end if;
    --
    -- Remove Company and Address_Name from address lines if they exist
    --
    if f_address_rec.Company IS NOT NULL then
      if    upper(NVL(f_address_rec.address_line_1,'x')) = upper(f_address_rec.Company) then
        f_address_rec.address_line_1 := f_address_rec.address_line_2;
        f_address_rec.address_line_2 := f_address_rec.address_line_3;
        f_address_rec.address_line_3 := NULL;
      elsif upper(NVL(f_address_rec.address_line_2,'x')) = upper(f_address_rec.Company) then
        f_address_rec.address_line_2 := f_address_rec.address_line_3;
        f_address_rec.address_line_3 := NULL;
      end if;
    end if;
    if f_address_rec.Address_Name IS NOT NULL then
      if    upper(NVL(f_address_rec.address_line_1,'x')) = upper(f_address_rec.Address_Name) then
        f_address_rec.address_line_1 := f_address_rec.address_line_2;
        f_address_rec.address_line_2 := f_address_rec.address_line_3;
        f_address_rec.address_line_3 := NULL;
      elsif upper(NVL(f_address_rec.address_line_2,'x')) = upper(f_address_rec.Address_Name) then
        f_address_rec.address_line_2 := f_address_rec.address_line_3;
        f_address_rec.address_line_3 := NULL;
      end if;
    end if;
    -- Make sure Company and Address_Name are not both set
    if     f_address_rec.Company IS NOT NULL
       and f_address_rec.Address_Name IS NOT NULL
       and upper(f_address_rec.Company) = upper(f_address_rec.Address_Name)
       then
      if lb_input_Address_Name then
        f_address_rec.Company := NULL;
      else
        f_address_rec.Address_Name := NULL;
      end if;
    end if;
    --
    -- Append the ZIP+4 if returned
    --
    if f_address_rec.POSTAL_CODE_EXTENDED IS NOT NULL
       -- Only use Zip+4 if the parameter is set to use it
       and gb_Use_ZIP_Plus4 then
      f_address_rec.POSTAL_CODE := f_address_rec.POSTAL_CODE||'-'||f_address_rec.POSTAL_CODE_EXTENDED;
    end if;
    --
    -- Fix International Addresses where Address 1-4 contains city, province, postal code, country
    --
    if NVL(f_address_rec.Status_Code,'x') like 'AD%' then
      --
      -- GLOBAL Address
      --
      -- Only use new address if we have a valid city or postal_code
      if    f_address_rec.city IS NOT NULL
         or f_address_rec.postal_code IS NOT NULL
         then
        if f_address_rec.address IS NOT NULL then
          f_address_rec.address_line_1 := f_address_rec.address;
          f_address_rec.address_line_2 := f_address_rec.suite;
          f_address_rec.address_line_3 := NULL;
          f_address_rec.address_line_4 := NULL;
        end if;
        -- Append State to City for international addresses if state field is to large and was not found in the STATE table - except for Canada
        if NVL(length(NVL(f_address_rec.State, f_address_rec.Province)),0) > gn_Max_State_Length
           and upper(NVL(f_address_rec.country_name,'x')) != 'CANADA' then
          if length(f_address_rec.city||', '||NVL(f_address_rec.State, f_address_rec.Province)) <= gn_Max_City_Length then
            -- try to append state to city field if it will fit
            f_address_rec.city := f_address_rec.city||', '||NVL(f_address_rec.State, f_address_rec.Province);
            f_address_rec.state := NULL;
            f_address_rec.province := NULL;
          elsif f_address_rec.address_line_2 IS NULL
            and length(NVL(f_address_rec.State, f_address_rec.Province)) <= gn_Max_Address_Length
            and NVL(upper(f_address_rec.suite),'x') != upper(NVL(f_address_rec.State, f_address_rec.Province)) then
            f_address_rec.address_line_2 := NVL(f_address_rec.State, f_address_rec.Province);
            f_address_rec.state := NULL;
            f_address_rec.province := NULL;
          elsif f_address_rec.address_line_3 IS NULL
            and length(NVL(f_address_rec.State, f_address_rec.Province)) <= gn_Max_Address_Length
            and NVL(upper(f_address_rec.suite),'x') != upper(NVL(f_address_rec.State, f_address_rec.Province)) then
            f_address_rec.address_line_3 := NVL(f_address_rec.State, f_address_rec.Province);
            f_address_rec.state := NULL;
            f_address_rec.province := NULL;
          else
            -- default state to NULL if it won't fit any of above fields
            f_address_rec.state := NULL;
            f_address_rec.province := NULL;
          end if;
        end if;
      end if;
    else
      -- US / Canada Address - don't return last line
      f_address_rec.Last_Line := NULL;
    end if;
    --
    -- Replace # sign with Apt on address lines because # confuses downstream printing applications
    -- Make sure address does not have APT already in it
    -- And only if this is not a "Unique" ZIP Code - ZIP+4 = 0001
    --
    if NVL(f_address_rec.Postal_Code_Extended,'0001') != '0001' then
      if instr(f_address_rec.address_line_1, '#') > 0 then
        if instr(upper(f_address_rec.address_line_1), 'APT') = 0 then
          f_address_rec.address_line_1 := replace(f_address_rec.address_line_1, '#', 'Apt');
          f_address_rec.suite          := replace(f_address_rec.suite, '#', 'Apt');
        else
          f_address_rec.address_line_1 := replace(replace(f_address_rec.address_line_1, '#', ' '), '  ', ' ');
          f_address_rec.suite          := trim(replace(replace(f_address_rec.suite, '#', ' '), '  ', ' '));
        end if;
      end if;
      if instr(f_address_rec.address_line_2, '#') > 0 then
        if instr(upper(f_address_rec.address_line_2), ' APT') = 0 then
          f_address_rec.address_line_2 := replace(f_address_rec.address_line_2, '#', 'Apt');
          f_address_rec.suite          := replace(f_address_rec.suite, '#', 'Apt');
        else
          f_address_rec.address_line_2 := replace(replace(f_address_rec.address_line_2, '#', ' '), '  ', ' ');
          f_address_rec.suite          := trim(replace(replace(f_address_rec.suite, '#', ' '), '  ', ' '));
        end if;
      end if;
      if instr(f_address_rec.address_line_3, '#') > 0 then
        if instr(upper(f_address_rec.address_line_3), 'APT') = 0 then
          f_address_rec.address_line_3 := replace(f_address_rec.address_line_3, '#', 'Apt');
          f_address_rec.suite          := replace(f_address_rec.suite, '#', 'Apt');
        else
          f_address_rec.address_line_3 := replace(replace(f_address_rec.address_line_3, '#', ' '), '  ', ' ');
          f_address_rec.suite          := trim(replace(replace(f_address_rec.suite, '#', ' '), '  ', ' '));
        end if;
      end if;
      if instr(f_address_rec.address_line_4, '#') > 0 then
        if instr(upper(f_address_rec.address_line_4), 'APT') = 0 then
          f_address_rec.address_line_4 := replace(f_address_rec.address_line_4, '#', 'Apt');
          f_address_rec.suite          := replace(f_address_rec.suite, '#', 'Apt');
        else
          f_address_rec.address_line_4 := replace(replace(f_address_rec.address_line_4, '#', ' '), '  ', ' ');
          f_address_rec.suite          := trim(replace(replace(f_address_rec.suite, '#', ' '), '  ', ' '));
        end if;
      end if;
    end if;
    --
    -- Truncate the Address Lines if they are too long - avoid database error
    -- - Truncate rather than wrap to avoid directionals being confused as apartment numbers
    --
    Trim_Address_Line(f_address_rec.address_line_1);
    Trim_Address_Line(f_address_rec.address_line_2);
    Trim_Address_Line(f_address_rec.address_line_3);
    Trim_Address_Line(f_address_rec.address_line_4);
    Trim_Address_Line(f_address_rec.address);
    --
    -- See if we should expand the street suffix
    --
    lv_parsed_address := NULL;
    if gb_expand_street_suffix
       and f_address_rec.Parsed_Street_Suffix IS NOT NULL
       and length(f_address_rec.address) <= gn_Max_Address_Length
       then
      --
      -- Expand the street suffix into a temporary address line
      --
      lv_parsed_address := CLEAN_Address.Expand_Street_Suffix (
                                 fv_address       => f_address_rec.address
                                ,fv_street_suffix => f_address_rec.Parsed_Street_Suffix
                                );
    end if;
    --
    -- Suite / Apartment Business Rules
    --
    if f_address_rec.suite IS NOT NULL
       -- 5/12/08 - don't append suite for international addresses
       and NVL(f_address_rec.Status_Code,'x') NOT like 'AD%' then
      --
      -- Check global parameter to see if we should append suite/apt to address line
      --
      if     f_address_rec.suite not like 'PO %'
         and f_address_rec.Address not like 'PO %'
         -- 6/18/09 - Make sure Unit exists before appending suite
         and f_address_rec.Parsed_Unit_Type IS NOT NULL
         and (   gb_Append_Suite_To_Street
         -- Always append suite if we have extra address fields in Company AND Address_Name that
         -- we need to add back in
         or (    f_address_rec.Company IS NOT NULL
             and NOT lb_company_input
             and f_address_rec.Address_Name IS NOT NULL
            )
         or (    f_address_rec.Attention IS NOT NULL
             and lb_company_input
             and f_address_rec.Address_Name IS NOT NULL
            )
             or (    f_address_rec.Company IS NOT NULL
                 and NOT lb_company_input
                 and f_address_rec.Attention IS NOT NULL
                )
            )
         then
        --------------------------------------------------
        -- PARSED ADDRESS - STREET SUFFIX EXPANSION
        --
        -- Put the suite on the same line as the address if there is room - and suite is on line 2
        --
        if f_address_rec.suite = f_address_rec.address_line_2
           and lv_parsed_address IS NOT NULL
           and length(lv_parsed_address||' '||f_address_rec.address_line_2) <= gn_Max_Address_Length
           then
          f_address_rec.address_line_1 := lv_parsed_address||' '||f_address_rec.address_line_2;
          f_address_rec.address_line_2 := f_address_rec.address_line_3;
          f_address_rec.address_line_3 := NULL;
          f_address_rec.address_line_4 := NULL;
          lb_suite_appended := TRUE;
          f_address_rec.address := lv_parsed_address;
          lv_parsed_address := NULL;
        end if;
        --
        -- Put the suite on the same line as the address if there is room - and suite is on line 3
        --
        if f_address_rec.suite = f_address_rec.address_line_3
           and lv_parsed_address IS NOT NULL
           and length(lv_parsed_address||' '||f_address_rec.address_line_3) <= gn_Max_Address_Length
         then
          f_address_rec.address_line_2 := f_address_rec.address_line_2||' '||f_address_rec.address_line_3;
          f_address_rec.address_line_3 := NULL;
          f_address_rec.address_line_4 := NULL;
          lb_suite_appended := TRUE;
          f_address_rec.address := lv_parsed_address;
          lv_parsed_address := NULL;
        end if;
        --------------------------------------------------
        -- NON-PARSED ADDRESS - NO STREET SUFFIX EXPANSION
        --
        -- Put the suite on the same line as the address if there is room - and suite is on line 2
        --
        if f_address_rec.suite = f_address_rec.address_line_2
           and length(f_address_rec.address_line_1||' '||f_address_rec.address_line_2) <= gn_Max_Address_Length
           then
          f_address_rec.address_line_1 := f_address_rec.address_line_1||' '||f_address_rec.address_line_2;
          f_address_rec.address_line_2 := f_address_rec.address_line_3;
          f_address_rec.address_line_3 := NULL;
          f_address_rec.address_line_4 := NULL;
          lb_suite_appended := TRUE;
        end if;
        --
        -- Put the suite on the same line as the address if there is room - and suite is on line 3
        --
        if f_address_rec.suite = f_address_rec.address_line_3
           and length(f_address_rec.address_line_2||' '||f_address_rec.address_line_3) <= gn_Max_Address_Length
           then
          f_address_rec.address_line_2 := f_address_rec.address_line_2||' '||f_address_rec.address_line_3;
          f_address_rec.address_line_3 := NULL;
          f_address_rec.address_line_4 := NULL;
          lb_suite_appended := TRUE;
        end if;
      end if;
      --
      -- Make sure the parsed address line is <= gn_Max_Address_Length characters
      --
      if lv_parsed_address IS NOT NULL
         and length(lv_parsed_address) <= gn_Max_Address_Length then
        if    f_address_rec.address_line_1 = f_address_rec.address then
          f_address_rec.address_line_1 := lv_parsed_address;
        elsif f_address_rec.address_line_2 = f_address_rec.address then
          f_address_rec.address_line_2 := lv_parsed_address;
        elsif f_address_rec.address_line_3 = f_address_rec.address then
          f_address_rec.address_line_3 := lv_parsed_address;
        elsif f_address_rec.address_line_4 = f_address_rec.address then
          f_address_rec.address_line_4 := lv_parsed_address;
        end if;
        f_address_rec.address := lv_parsed_address;
        lv_parsed_address := NULL;
      end if;
      --
      -- See if we should switch the suite/apartment line above the address line
      --
      if    (    gb_suite_apt_before_street
             -- 7/10/09 - Make sure Unit exists before adjusting suite
             and f_address_rec.Parsed_Unit_Type IS NOT NULL
             and NOT lb_suite_appended
            )
         -- 7/31/09 Always put address after any non-suite field - i.e. attention above address - except for unique zipcodes 0001
         or (    f_address_rec.Parsed_Unit_Type IS NULL
             and f_address_rec.suite IS NOT NULL
             and f_address_rec.address IS NOT NULL
             and f_address_rec.postal_code_extended != '0001'
             and f_address_rec.postal_code_extended != '0002'
             and NOT lb_suite_appended
            )
         then
        --
        -- Put the suite on line 1 if it is on line 2
        --
        if f_address_rec.address = f_address_rec.address_line_1
           and f_address_rec.suite = f_address_rec.address_line_2
           then
          f_address_rec.address_line_1 := f_address_rec.suite;
          f_address_rec.address_line_2 := f_address_rec.address;
        end if;
        --
        -- Put the suite on line 2 if it is on line 3
        --
        if f_address_rec.address = f_address_rec.address_line_2
           and f_address_rec.suite = f_address_rec.address_line_3
           then
          f_address_rec.address_line_2 := f_address_rec.suite;
          f_address_rec.address_line_3 := f_address_rec.address;
        end if;
        --
        -- Put the suite on line 3 if it is on line 4
        --
        if f_address_rec.address = f_address_rec.address_line_3
           and f_address_rec.suite = f_address_rec.address_line_4
           then
          f_address_rec.address_line_3 := f_address_rec.suite;
          f_address_rec.address_line_4 := f_address_rec.address;
        end if;
      end if;
    end if;
    --
    -- Handle the expanded street suffix buffer when there is no suite
    -- Make sure the parsed address line is <= gn_Max_Address_Length characters
    --
    if lv_parsed_address IS NOT NULL
       and length(lv_parsed_address) <= gn_Max_Address_Length then
      if    f_address_rec.address_line_1 = f_address_rec.address then
        f_address_rec.address_line_1 := lv_parsed_address;
      elsif f_address_rec.address_line_2 = f_address_rec.address then
        f_address_rec.address_line_2 := lv_parsed_address;
      elsif f_address_rec.address_line_3 = f_address_rec.address then
        f_address_rec.address_line_3 := lv_parsed_address;
      elsif f_address_rec.address_line_4 = f_address_rec.address then
        f_address_rec.address_line_4 := lv_parsed_address;
      end if;
      f_address_rec.address := lv_parsed_address;
      lv_parsed_address := NULL;
    end if;
    --
    -- Add Attention and/or Company back on to address if it was parsed out of address line 3
    -- - Could be stored in company or attention
    --
    if f_address_rec.attention||f_address_rec.address_name||f_address_rec.company IS NOT NULL
       and f_address_rec.address_line_3 IS NULL
       then
      -- make sure Company is not already listed on line 1
      if f_address_rec.company IS NOT NULL
         and upper(f_address_rec.company) != upper(NVL(f_address_rec.address_line_1, ' '))
         and upper(f_address_rec.company) != upper(NVL(f_address_rec.address_line_2, ' '))
         and f_address_rec.address_line_3 IS NULL
         and NOT lb_company_input  -- don't append company if it was input separately (Finance)
         then
        f_address_rec.address_line_3 := f_address_rec.address_line_2;
        f_address_rec.address_line_2 := f_address_rec.address_line_1;
        f_address_rec.address_line_1 := f_address_rec.company;
        f_address_rec.company        := NULL;
      end if;
      -- make sure Attention is not already listed on line 1
      if f_address_rec.attention IS NOT NULL
         and upper(f_address_rec.attention) != upper(NVL(f_address_rec.address_line_1, ' '))
         and upper(f_address_rec.attention) != upper(NVL(f_address_rec.address_line_2, ' '))
         and f_address_rec.address_line_3 IS NULL
         then
        f_address_rec.address_line_3 := f_address_rec.address_line_2;
        f_address_rec.address_line_2 := f_address_rec.address_line_1;
        f_address_rec.address_line_1 := f_address_rec.attention;
        f_address_rec.attention      := NULL;
      end if;
      -- make sure Address_Name is not already listed on line 1
      if f_address_rec.Address_Name IS NOT NULL
         and upper(f_address_rec.Address_Name) != upper(NVL(f_address_rec.address_line_1, ' '))
         and upper(f_address_rec.Address_Name) != upper(NVL(f_address_rec.address_line_2, ' '))
         and f_address_rec.address_line_3 IS NULL
         then
        f_address_rec.address_line_3 := f_address_rec.address_line_2;
        f_address_rec.address_line_2 := f_address_rec.address_line_1;
        f_address_rec.address_line_1 := f_address_rec.Address_Name;
        f_address_rec.Address_Name   := NULL;
      end if;
      end if;
    --
    -- 7/10/09 - account for address_line_4
    -- Add Attention and/or Company back on to address if it was parsed out of address line 3
    -- - Could be stored in company or attention
    --
    if f_address_rec.attention||f_address_rec.address_name||f_address_rec.company IS NOT NULL
       and f_address_rec.address_line_4 IS NULL
       then
      -- make sure Company is not already listed on line 1
      if f_address_rec.company IS NOT NULL
         and upper(f_address_rec.company) != upper(NVL(f_address_rec.address_line_1, ' '))
         and upper(f_address_rec.company) != upper(NVL(f_address_rec.address_line_2, ' '))
         and f_address_rec.address_line_4 IS NULL
         and NOT lb_company_input  -- don't append company if it was input separately (Finance)
         then
        f_address_rec.address_line_4 := f_address_rec.address_line_3;
        f_address_rec.address_line_3 := f_address_rec.address_line_2;
        f_address_rec.address_line_2 := f_address_rec.address_line_1;
        f_address_rec.address_line_1 := f_address_rec.company;
        f_address_rec.company        := NULL;
      end if;
      -- make sure Attention is not already listed on line 1
      if f_address_rec.attention IS NOT NULL
         and upper(f_address_rec.attention) != upper(NVL(f_address_rec.address_line_1, ' '))
         and upper(f_address_rec.attention) != upper(NVL(f_address_rec.address_line_2, ' '))
         and f_address_rec.address_line_4 IS NULL
         then
        f_address_rec.address_line_4 := f_address_rec.address_line_3;
        f_address_rec.address_line_3 := f_address_rec.address_line_2;
        f_address_rec.address_line_2 := f_address_rec.address_line_1;
        f_address_rec.address_line_1 := f_address_rec.attention;
        f_address_rec.attention      := NULL;
      end if;
      -- make sure Address_Name is not already listed on line 1
      if f_address_rec.Address_Name IS NOT NULL
         and upper(f_address_rec.Address_Name) != upper(NVL(f_address_rec.address_line_1, ' '))
         and upper(f_address_rec.Address_Name) != upper(NVL(f_address_rec.address_line_2, ' '))
         and f_address_rec.address_line_4 IS NULL
         then
        f_address_rec.address_line_4 := f_address_rec.address_line_3;
        f_address_rec.address_line_3 := f_address_rec.address_line_2;
        f_address_rec.address_line_2 := f_address_rec.address_line_1;
        f_address_rec.address_line_1 := f_address_rec.Address_Name;
        f_address_rec.Address_Name   := NULL;
      end if;
    end if;
    --
    -- make sure city does not go over gn_Max_City_Length characters - else use City abbreviation
    --
    if length(f_address_rec.city) > gn_Max_City_Length then
      f_address_rec.city := NVL(substr(f_address_rec.city_abbreviation,1,gn_Max_City_Length), substr(f_address_rec.city,1,gn_Max_City_Length));
    end if;
    --
    -- Check for DPV error and use this if the error code is null
    --
    if NVL(f_address_rec.DPV_Status,'AA') NOT IN ('AA','BB','RR')
       and f_address_rec.error_code IS NULL then
      -- Use DPV Status for Error Code
      f_address_rec.error_code   := f_address_rec.DPV_Status;
      f_address_rec.error_string := NVL(cln$lookup.Help_DPV_Status(f_address_rec.DPV_Status)
                                       ,cln$lookup.Get_DPV_Status(f_address_rec.DPV_Status)
                                       );
      -- Append the address suggestion list to the end of the error string - limit to 500 characters
      if f_address_rec.suggestion_list IS NOT NULL then
        f_address_rec.error_string := substr(f_address_rec.error_string||chr(10)||'Suggestions:'||chr(10)||replace(f_address_rec.suggestion_list,';',chr(10)),1,500);
      end if;
    --
    -- get the address status code for the error code for international addresses
    --
    elsif NVL(f_address_rec.Status_Code,'x') like 'AD%' then
      BEGIN
        -- only use address status if the error code is a number
        if to_number(replace(f_address_rec.error_code,'%')) >= 0 then
          --
          -- Check for Valid International Address
          --
          if NVL(f_address_rec.Status_Code,'ADC') IN ('ADC','ADV') then
            f_address_rec.error_code   := NULL;
            f_address_rec.error_string := NULL;
          else
            f_address_rec.error_code   := f_address_rec.Status_Code;
            f_address_rec.error_string := cln$lookup.Get_Address_Status(f_address_rec.Status_Code);
          end if;
        end if;
      EXCEPTION
        WHEN OTHERS THEN
          NULL;
      END;
    elsif f_address_rec.error_code IS NOT NULL then
      f_address_rec.error_string := NVL(cln$lookup.Get_Address_Error(f_address_rec.Error_Code)
                                       ,f_address_rec.error_string
                                       );
      -- Append the address suggestion list to the end of the error string - limit to 500 characters
      if f_address_rec.suggestion_list IS NOT NULL then
        f_address_rec.error_string := substr(f_address_rec.error_string||chr(10)||'Suggestions:'||chr(10)||replace(f_address_rec.suggestion_list,';',chr(10)),1,500);
      end if;
    end if;
    --
    -- Support for up to 7 address lines - Restore previously backed up address lines toward the top
    --
    if trim(l_address_rec_bak.Address_Line_7) IS NOT NULL then
      f_address_rec.Address_Line_7 := f_address_rec.Address_Line_4;
      f_address_rec.Address_Line_6 := f_address_rec.Address_Line_3;
      f_address_rec.Address_Line_5 := f_address_rec.Address_Line_2;
      f_address_rec.Address_Line_4 := f_address_rec.Address_Line_1;
      f_address_rec.Address_Line_1 := l_address_rec_bak.Address_Line_1;
      f_address_rec.Address_Line_2 := l_address_rec_bak.Address_Line_2;
      f_address_rec.Address_Line_3 := l_address_rec_bak.Address_Line_3;
   elsif trim(l_address_rec_bak.Address_Line_6) IS NOT NULL then
      f_address_rec.Address_Line_6 := f_address_rec.Address_Line_4;
      f_address_rec.Address_Line_5 := f_address_rec.Address_Line_3;
      f_address_rec.Address_Line_4 := f_address_rec.Address_Line_2;
      f_address_rec.Address_Line_3 := f_address_rec.Address_Line_1;
      f_address_rec.Address_Line_1 := l_address_rec_bak.Address_Line_1;
      f_address_rec.Address_Line_2 := l_address_rec_bak.Address_Line_2;
    elsif trim(l_address_rec_bak.Address_Line_5) IS NOT NULL then
      f_address_rec.Address_Line_5 := f_address_rec.Address_Line_4;
      f_address_rec.Address_Line_4 := f_address_rec.Address_Line_3;
      f_address_rec.Address_Line_3 := f_address_rec.Address_Line_2;
      f_address_rec.Address_Line_2 := f_address_rec.Address_Line_1;
      f_address_rec.Address_Line_1 := l_address_rec_bak.Address_Line_1;
    end if;
    --
    -- Get the address_verified_date field - use Address_Line_8 for the buffer
    --
    -- address_line_8 = [Verified_Date]
    f_address_rec.address_line_8 := to_char(sysdate, 'YYYYMMDD');
  EXCEPTION
    WHEN OTHERS THEN
      fv_plsql_error := SQLERRM;
  END;
  --
  -- Post Address Process User Exit
  -- - Make any custom changes in the CLEAN_Address_APP_UE User Exit package
  --
  CLEAN_Address_APP_UE.Post_Address_Process(
         f_address_rec     => f_address_rec
        ,fv_plsql_error    => fv_plsql_error
        ,fv_address_type   => fv_address_type
        );
  --
  -- Standardize the Application fields to fit in Application table structure
  --
  f_address_rec.Address_Name       := NVL(NVL(f_address_rec.Address_Name, f_address_rec.Attention),' ');
  f_address_rec.Address_Line_1     := NVL(substr(f_address_rec.Address_Line_1, 1, gn_Max_Address_Length),' ');
  f_address_rec.Address_Line_2     := NVL(substr(f_address_rec.Address_Line_2, 1, gn_Max_Address_Length),' ');
  f_address_rec.Address_Line_3     := NVL(substr(f_address_rec.Address_Line_3, 1, gn_Max_Address_Length),' ');
  f_address_rec.Address_Line_4     := NVL(substr(f_address_rec.Address_Line_4, 1, gn_Max_Address_Length),' ');
  f_address_rec.City               := NVL(substr(f_address_rec.City,           1, gn_Max_City_Length),' ');
  f_address_rec.State              := NVL(substr(NVL(f_address_rec.State, f_address_rec.Province),1,3),' ');
  f_address_rec.Postal_Code        := NVL(f_address_rec.Postal_Code,' ');
  f_address_rec.County_Name        := NVL(f_address_rec.County_Name,' ');
  -- Address_Line_7 = Complete POSTNET_BARCODE
  f_address_rec.Address_Line_7     := replace(f_address_rec.Postal_code,'-')||f_address_rec.delivery_point||f_address_rec.check_digit;
  f_address_rec.Last_Line          := substr(f_address_rec.Last_Line,1,55);  -- City, State ZIP Line
END Verify_CLEAN_Address_Record;
/******************************************************************************************
 *  Procedure Name  :   Check_Telephone_Record
 *
 *  Scope           :   PUBLIC
 *
 *  Description     :   Verify a single telephone number for Application
 *                      - accepts CLEAN_Address Telephone record type as input
 ******************************************************************************************/
PROCEDURE Check_Telephone_Record (
                     f_telephone_rec                IN OUT CLEAN_Address.Phone_rec
                    ,fv_phone_area                  IN     VARCHAR2
                    ,fv_plsql_error                 IN OUT VARCHAR2
                    ,fv_telephone_code              IN     VARCHAR2 DEFAULT NULL
                    ,fv_object_name                 IN     VARCHAR2 DEFAULT NULL /* Object Name where procedure is called from */
                    )
IS
  lb_verified         BOOLEAN       := FALSE;
BEGIN
  fv_plsql_error := NULL;
  -- 3/6/08 - Set the object name so user exit procedure can identify where the call was made from
  Set_Object_Name(fv_object_name);
  --
  -- Verify the Telephone
  --
  BEGIN
    -- Default the phone output format
    f_telephone_rec.Phone_Number_Output_Format := '999-999-9999';
    --
    -- Pre Telephone Check User Exit
    -- - Make any custom changes in the CLEAN_Address_APP_UE User Exit package
    --
    CLEAN_Address_APP_UE.Pre_Telephone_Check(
             f_telephone_rec   => f_telephone_rec
            ,fv_plsql_error    => fv_plsql_error
            ,fv_telephone_type => fv_telephone_code
            ,fb_verified       => lb_verified
            );
    --
    -- Exit if the address was verified or overridden in the User Exit
    --
    if lb_verified then
      return;
    end if;
    --
    -- Process an Address with Standardized component fields
    --
    CLEAN_Address.Phone_Check(
           f_phone_rec => f_telephone_rec
          );
    --
    -- Post Telephone Check User Exit
    -- - Make any custom changes in the CLEAN_Address_APP_UE User Exit package
    --
    CLEAN_Address_APP_UE.Post_Telephone_Check(
             f_telephone_rec   => f_telephone_rec
            ,fv_plsql_error    => fv_plsql_error
            ,fv_telephone_type => fv_telephone_code
            ,fb_verified       => lb_verified
            );
    --
    -- Exit if the address was verified or overridden in the User Exit
    --
    if lb_verified then
      return;
    end if;
    --
    -- Get the suggested area code and distance
    --
    -- Assign the area code back only if this was an area code SPLIT
    if f_telephone_rec.Phone_Check_Status_Code = 'U'
       or fv_phone_area IS NULL then
      -- override the Phone area if there was a split of no area on input
      f_telephone_rec.Phone_Area_Code := f_telephone_rec.New_Phone_Area_Code;
      f_telephone_rec.Phone_Number    := f_telephone_rec.New_Phone_Number;
    else
      -- Use original phone area if we didn't have an area code split and it was present before
      f_telephone_rec.Phone_Area_Code := NVL(fv_phone_area, f_telephone_rec.Phone_Area_Code);
    end if;
    -- Only get suggested area code for certain status
    --  C - corrected area code based on Zip if falls into
    if NVL(f_telephone_rec.Phone_Check_Status_Code,'x') != 'C' then
      f_telephone_rec.New_Phone_Area_Code := NULL;
    end if;
    -- Ignore pseudo distances (800 and cell phone numbers)
    if NVL(f_telephone_rec.Phone_Distance, 9999) = 9999 then
      f_telephone_rec.Phone_Distance := NULL;
    end if;
    -- Only accept certain error codes
    --  A - Bad Area Code
    --  B - Phone number is Blank
    --  E - Bad Phone Number - too many or too few digits
    --  P - Bad Prefix
    if NVL(f_telephone_rec.Phone_Check_Error_Code, 'x') not in ('A','B','E','P') then
      f_telephone_rec.Phone_Check_Error_Code := NULL;
    end if;
  EXCEPTION
    WHEN OTHERS THEN
      fv_plsql_error := SQLERRM;
  END;
  --
  -- Post Telephone Check User Exit
  -- - Make any custom changes in the CLEAN_Address_APP_UE User Exit package
  --
  CLEAN_Address_APP_UE.Post_Telephone_Process(
           f_telephone_rec   => f_telephone_rec
          ,fv_plsql_error    => fv_plsql_error
          ,fv_telephone_type => fv_telephone_code
          );
END Check_Telephone_Record;
/******************************************************************************************
 *  Procedure Name  :   Standardize_Name_Record
 *
 *  Scope           :   PUBLIC
 *
 *  Description     :   Standardize a single name record for Application
 *                      - accepts CLEAN_Address Name record type as input
 ******************************************************************************************/
PROCEDURE Standardize_Name_Record (
                     f_name_rec                     IN OUT CLEAN_Address.Name_rec
                    ,fv_name_error_text             IN OUT VARCHAR2
                    ,fv_plsql_error                 IN OUT VARCHAR2
                    ,fv_name_type_code              IN     VARCHAR2 DEFAULT NULL
                    ,fv_object_name                 IN     VARCHAR2 DEFAULT NULL /* Object Name where procedure is called from */
                    )
IS
  lb_verified         BOOLEAN       := FALSE;
  l_name_rec_bak      CLEAN_Address.Name_rec;
BEGIN
  fv_plsql_error := NULL;
  fv_name_error_text := NULL;
  -- Set the object name so user exit procedure can identify where the call was made from
  Set_Object_Name(fv_object_name);
  -- Backup the name record in case we get unknown first name
  l_name_rec_bak := f_name_rec;
  --
  -- Standardize the Name
  --
  BEGIN
    -- Build the full name field
    f_name_rec.full_name := trim(     f_name_rec.name_prefix
                               ||' '||f_name_rec.first_name
                               ||' '||f_name_rec.middle_name
                               ||' '||f_name_rec.last_name
                               ||' '||f_name_rec.name_suffix
                               );
    --
    -- Pre Name Standardize User Exit
    -- - Make any custom changes in the CLEAN_Address_APP_UE User Exit package
    --
    CLEAN_Address_APP_UE.Pre_Name_Standardize(
             f_name_rec         => f_name_rec
            ,fv_name_error_text => fv_name_error_text
            ,fv_plsql_error     => fv_plsql_error
            ,fv_name_type_code  => fv_name_type_code
            ,fb_verified        => lb_verified
            );
    --
    -- Exit if the record was verified or overridden in the User Exit
    --
    if lb_verified then
      return;
    end if;
    --
    -- Process a Name record
    --
    CLEAN_Address.Name_Parse(
           f_name_rec => f_name_rec
          );
    -- Check for other name errors
    if f_name_rec.Name_Parse_Status_Code IS NOT NULL then
      fv_name_error_text := Get_Name_Error_Text(f_name_rec.Name_Parse_Status_Code);
    end if;
    -- Check for Unknown Gender - possible misspelling of first name
    if f_name_rec.Gender = 'U' then
      f_name_rec.Name_Parse_Status_Code := 'U';
      fv_name_error_text := Get_Name_Error_Text(f_name_rec.Gender);
    end if;
    --  Restore the original name fields if we got an error message
    if f_name_rec.Name_Parse_Status_Code IS NOT NULL then
      f_name_rec.name_prefix := clean_address.InitName(l_name_rec_bak.name_prefix);
      f_name_rec.first_name  := clean_address.InitName(l_name_rec_bak.first_name);
      f_name_rec.middle_name := clean_address.InitName(l_name_rec_bak.middle_name);
      f_name_rec.last_name   := clean_address.InitName(l_name_rec_bak.last_name);
      f_name_rec.name_suffix := clean_address.InitName(l_name_rec_bak.name_suffix);
    end if;
    --
    -- Post Name Standardize User Exit
    -- - Make any custom changes in the CLEAN_Address_APP_UE User Exit package
    --
    CLEAN_Address_APP_UE.Post_Name_Standardize(
             f_name_rec         => f_name_rec
            ,fv_name_error_text => fv_name_error_text
            ,fv_plsql_error     => fv_plsql_error
            ,fv_name_type_code  => fv_name_type_code
            ,fb_verified        => lb_verified
            );
    --
    -- Exit if the record was verified or overridden in the User Exit
    --
    if lb_verified then
      return;
    end if;
  EXCEPTION
    WHEN OTHERS THEN
      fv_plsql_error := SQLERRM;
  END;
END Standardize_Name_Record;
/******************************************************************************************
 *  Procedure Name  :   Validate_Email_Record
 *
 *  Scope           :   PUBLIC
 *
 *  Description     :   Standardize a single name record for Application
 *                      - accepts CLEAN_Address Name record type as input
 ******************************************************************************************/
PROCEDURE Validate_Email_Record (
                     f_email_rec                    IN OUT CLEAN_Address.Email_rec
                    ,fv_email_error_text            IN OUT VARCHAR2
                    ,fv_plsql_error                 IN OUT VARCHAR2
                    ,fv_email_type_code             IN     VARCHAR2 DEFAULT NULL /* GOREMAL_EMAL_CODE */
                    ,fv_object_name                 IN     VARCHAR2 DEFAULT NULL /* Object Name where procedure is called from */
                    )
IS
  lb_verified         BOOLEAN       := FALSE;
  lb_dns_lookup       BOOLEAN       := TRUE;
BEGIN
  fv_plsql_error := NULL;
  fv_email_error_text := NULL;
  -- Set the object name so user exit procedure can identify where the call was made from
  Set_Object_Name(fv_object_name);
  --
  -- Validate the Email
  --
  BEGIN
    --
    -- Pre Email Validate User Exit
    -- - Make any custom changes in the CLEAN_Address_APP_UE User Exit package
    --
    CLEAN_Address_APP_UE.Pre_Email_Validate(
             f_email_rec         => f_email_rec
            ,fv_email_error_text => fv_email_error_text
            ,fv_plsql_error      => fv_plsql_error
            ,fv_email_type_code  => fv_email_type_code
            ,fb_dns_lookup       => lb_dns_lookup
            ,fb_verified         => lb_verified
            );
    --
    -- Exit if the record was verified or overridden in the User Exit
    --
    if lb_verified then
      return;
    end if;
    -- Standardize the email address first
    f_email_rec.email_address := replace(f_email_rec.email_address, ' ');
    --
    -- Process an Email record
    --
    -- See if we need to do the DNS lookup or just parse/standardize
    if lb_dns_lookup then
      CLEAN_Address.Email_Check(
           f_email_rec => f_email_rec
          );
    else
      CLEAN_Address.Email_Parse(
           f_email_rec => f_email_rec
          );
    end if;
    -- Get the Email Error Text
    fv_email_error_text := Get_Email_Error_Text(f_email_rec.email_status_code);
    --
    -- Post Email Validate User Exit
    -- - Make any custom changes in the CLEAN_Address_APP_UE User Exit package
    --
    CLEAN_Address_APP_UE.Post_Email_Validate(
             f_email_rec         => f_email_rec
            ,fv_email_error_text => fv_email_error_text
            ,fv_plsql_error      => fv_plsql_error
            ,fv_email_type_code  => fv_email_type_code
            ,fb_dns_lookup       => lb_dns_lookup
            ,fb_verified         => lb_verified
            );
    --
    -- Exit if the record was verified or overridden in the User Exit
    --
    if lb_verified then
      return;
    end if;
  EXCEPTION
    WHEN OTHERS THEN
      fv_plsql_error := SQLERRM;
  END;
END Validate_Email_Record;
/******************************************************************************************
 *  Procedure Name  :   Verify
 *
 *  Scope           :   PUBLIC
 *
 *  Description     :   Verify a single address for Application
 *
 *                      This call wraps the Address record implementation and excepts pure
 *                      VARCHAR2 IN/OUT parameters.
 ******************************************************************************************/
PROCEDURE Verify (
                     company_name                   IN OUT VARCHAR2 /* COMPANY */
                    ,address_1                      IN OUT VARCHAR2 /* ADDRESS1 */
                    ,address_2                      IN OUT VARCHAR2 /* ADDRESS2 */
                    ,address_3                      IN OUT VARCHAR2 /* ADDRESS3 */
                    ,address_4                      IN OUT VARCHAR2 /* ADDRESS4 */
                    ,city                           IN OUT VARCHAR2 /* CITY */
                    ,state                          IN OUT VARCHAR2 /* STATE */
                    ,postal_code                    IN OUT VARCHAR2 /* POSTAL */
                    ,county                         IN OUT VARCHAR2 /* COUNTY */
                    ,country_code                   IN OUT VARCHAR2 /* COUNTRY */
                    ,carrier_route                  IN OUT VARCHAR2 /* CLN_ADDRESS_APP.carrier_route */
                    ,delivery_point                 IN OUT VARCHAR2 /* CLN_ADDRESS_APP.delivery_point */
                    ,check_digit                    IN OUT VARCHAR2 /* CLN_ADDRESS_APP.check_digit */
                    ,postnet_barcode                IN OUT VARCHAR2 /* CLN_ADDRESS_APP.postnet_barcode */
                    ,last_line                      IN OUT VARCHAR2 /* CLN_ADDRESS_APP.last_line */
                    ,latitude                       IN OUT NUMBER   /* CLN_ADDRESS_APP.Latitude */
                    ,longitude                      IN OUT NUMBER   /* CLN_ADDRESS_APP.Longitude */
                    ,address_error_code             IN OUT VARCHAR2 /* CLN_ADDRESS_APP.ADDRESS_ERROR_CODE */
                    ,address_verified_date          IN OUT DATE     /* CLN_ADDRESS_APP.ADDRESS_VERIFIED_DATE */
                    ,suggestion_list                IN OUT VARCHAR2 /* CLN_ADDRESS_APP.suggestion_list */
                    ,address_error_text             IN OUT VARCHAR2
                    ,fv_plsql_error                 IN OUT VARCHAR2
                    ,fv_address_type                IN     VARCHAR2 DEFAULT NULL /* ADDRESS_TYPE */
                    ,fv_object_name                 IN     VARCHAR2 DEFAULT NULL /* Object Name where procedure is called from */
                    )
IS
  -- CLEAN_Address record structure
  l_address_rec       CLEAN_Address.Address_rec;
BEGIN
  --
  -- Assign the input variables to the address record
  --
  --l_address_rec.Address_Name   := trim(attention);
  l_address_rec.Company        := trim(company_name);
  l_address_rec.Address_Line_1 := trim(address_1);
  l_address_rec.Address_Line_2 := trim(address_2);
  l_address_rec.Address_Line_3 := trim(address_3);
  l_address_rec.Address_Line_4 := trim(address_4);
  l_address_rec.City           := trim(city);
  l_address_rec.State          := trim(state);
  l_address_rec.Postal_code    := trim(postal_code);
  l_address_rec.county_name    := trim(county);
  l_address_rec.country_code   := trim(country_code);
  --
  -- Verify the Address
  --
  Verify_CLEAN_Address_Record (
                f_address_rec   => l_address_rec
               ,fv_plsql_error  => fv_plsql_error
               ,fv_address_type => fv_address_type
               ,fv_object_name  => fv_object_name
               );
  --
  -- Assign the address record back to the output variables
  --
  company_name              := l_address_rec.Company;
  address_1                 := l_address_rec.Address_Line_1;
  address_2                 := l_address_rec.Address_Line_2;
  address_3                 := l_address_rec.Address_Line_3;
  address_4                 := l_address_rec.Address_Line_4;
  city                      := l_address_rec.City;
  state                     := l_address_rec.State;
  postal_code               := l_address_rec.Postal_code;
  county                    := l_address_rec.county_name;
  country_code              := l_address_rec.country_code;
  address_error_code        := l_address_rec.Error_Code;
  address_verified_date     := to_date(l_address_rec.Address_Line_8, 'YYYYMMDD');
  address_error_text        := l_address_rec.error_string;
  latitude                  := l_address_rec.latitude;
  longitude                 := l_address_rec.longitude;
  carrier_route             := l_address_rec.carrier_route;
  delivery_point            := l_address_rec.delivery_point;
  check_digit               := l_address_rec.check_digit;
  postnet_barcode           := l_address_rec.address_line_7; /* Postnet_barcode */
  last_line                 := l_address_rec.last_line;
  suggestion_list           := l_address_rec.suggestion_list;
END Verify;

/******************************************************************************************
 *  Procedure Name  :   Verify_Generic_Silent
 *
 *  Scope           :   PUBLIC
 *
 *  Description     :   Verify an address using Generic Address Lines 1-4
 *                      - returns the standardized and verified address components
 *                        as well as the new Address Lines 1-4
 ******************************************************************************************/
PROCEDURE Verify_Generic_Silent (
                     f_address_rec                  IN OUT CLEAN_Address.Address_rec
                    ,fn_number_address_lines        IN     NUMBER      DEFAULT 3
                    ,fb_generic_address_name        IN     BOOLEAN     DEFAULT FALSE
                    ,fb_generic_company             IN     BOOLEAN     DEFAULT FALSE
                    ,fb_address_suite_same_line     IN     BOOLEAN     DEFAULT FALSE
                    ,fv_return_code                    OUT VARCHAR2
                    ,fv_return_description             OUT VARCHAR2
                    ,fv_address_pipe_name           IN     VARCHAR2    DEFAULT NULL
                    ,fn_error_suggest_count         IN     PLS_INTEGER DEFAULT 0
                    ,fb_verbose_suggest_list        IN     BOOLEAN     DEFAULT FALSE
                    ,fb_upper_case_results          IN     BOOLEAN     DEFAULT FALSE
                    ,address_error_text             IN OUT VARCHAR2
                    ,fv_plsql_error                 IN OUT VARCHAR2
                    )
IS
  lv_error  VARCHAR2(500);
BEGIN

  CLEAN_ADDRESS.Verify_Generic(
         f_address_rec              => f_address_rec
        ,fn_number_address_lines    => fn_number_address_lines
        ,fb_generic_address_name    => fb_generic_address_name
        ,fb_generic_company         => fb_generic_company
        ,fb_address_suite_same_line => fb_address_suite_same_line
        ,fv_address_pipe_name       => fv_address_pipe_name
        ,fn_error_suggest_count     => 20
        ,fb_verbose_suggest_list    => fb_verbose_suggest_list
        ,fb_upper_case_results      => fb_upper_case_results
        );
    --
    -- Check for DPV error and use this if the error code is null
    --
    if NVL(f_address_rec.DPV_Status,'AA') NOT IN ('AA','BB','RR')
       and f_address_rec.error_code IS NULL
       then
      -- Use DPV Status for Error Code
      f_address_rec.error_code   := f_address_rec.DPV_Status;
      f_address_rec.error_string := NVL(cln$lookup.Help_DPV_Status(f_address_rec.DPV_Status)
                                       ,cln$lookup.Get_DPV_Status(f_address_rec.DPV_Status)
                                       );
      -- Append the address suggestion list to the end of the error string - limit to 500 characters
      if f_address_rec.suggestion_list IS NOT NULL then
        f_address_rec.error_string := substr(f_address_rec.error_string||chr(10)||'Suggestions:'||chr(10)||replace(f_address_rec.suggestion_list,';',chr(10)),1,500);
      end if;
    --
    -- get the address status code for the error code for international addresses
    --
    elsif NVL(f_address_rec.Status_Code,'x') like 'AD%' then
      BEGIN
        -- only use address status if the error code is a number
        if to_number(replace(f_address_rec.error_code,'%')) >= 0 then
          --
          -- Check for Valid International Address
          --
          if NVL(f_address_rec.Status_Code,'ADC') IN ('ADC','ADV') then
            f_address_rec.error_code   := NULL;
            f_address_rec.error_string := NULL;
          else
            f_address_rec.error_code   := f_address_rec.Status_Code;
            f_address_rec.error_string := cln$lookup.Get_Address_Status(f_address_rec.Status_Code);
          end if;
        end if;
      EXCEPTION
        WHEN OTHERS THEN
          NULL;
      END;
    elsif f_address_rec.error_code IS NOT NULL then
      f_address_rec.error_string := NVL(cln$lookup.Get_Address_Error(f_address_rec.Error_Code)
                                       ,f_address_rec.error_string
                                       );
      -- Append the address suggestion list to the end of the error string - limit to 500 characters
      if f_address_rec.suggestion_list IS NOT NULL then
        f_address_rec.error_string := substr(f_address_rec.error_string||chr(10)||'Suggestions:'||chr(10)||replace(f_address_rec.suggestion_list,';',chr(10)),1,500);
      end if;
    end if;
  address_error_text          := substr(f_address_rec.error_string,1,500);
  fv_plsql_error:=null;
  fv_return_code := NULL;
  fv_return_description := NULL;
EXCEPTION
  WHEN OTHERS THEN
    lv_error := replace(SQLERRM, 'ORA-20000: ');
    fv_return_code := substr(lv_error, 1, 9);
    fv_return_description := substr(lv_error, 11);
END Verify_Generic_Silent;

/******************************************************************************************
 *  Procedure Name  :   Verify_CLN_ADDRESSES_Record
 *
 *  Scope           :   PUBLIC
 *
 *  Description     :   Verify a single address by passing in the ADDRESS Record type
 ******************************************************************************************/
PROCEDURE Verify_CLN_ADDRESSES_Record (
                     f_ADDRESS_rec                  IN OUT CLEAN_Address.Address_rec
                    ,f_CLN_ADDRESS_APP_rec          IN OUT CLN_ADDRESS_APP%ROWTYPE
                    ,address_error_text             IN OUT VARCHAR2
                    ,fv_plsql_error                 IN OUT VARCHAR2
                    ,fv_company                     IN     VARCHAR2 DEFAULT NULL /* Company name can return different ZIP+4 sometimes */
                    ,fv_address_type                IN     VARCHAR2 DEFAULT NULL /* ADDRESS_TYPE */
                    ,fv_object_name                 IN     VARCHAR2 DEFAULT NULL /* Object Name where procedure is called from */
                    )
IS
  lv_company VARCHAR2(200);
BEGIN
  --
  -- Call the CLEAN_Address Application Verify procedure with individual parameters
  --
  lv_company := fv_company;
  CLEAN_Address_APP.Verify (
    	     company_name                => lv_company
    	    ,address_1                   => f_ADDRESS_rec.ADDRESS_LINE_1
    	    ,address_2                   => f_ADDRESS_rec.ADDRESS_LINE_2
    	    ,address_3                   => f_ADDRESS_rec.ADDRESS_LINE_3
    	    ,address_4                   => f_ADDRESS_rec.ADDRESS_LINE_4
    	    ,city                        => f_ADDRESS_rec.CITY
    	    ,state                       => f_ADDRESS_rec.STATE
    	    ,postal_code                 => f_ADDRESS_rec.POSTAL_CODE
    	    ,county                      => f_ADDRESS_rec.COUNTY_NAME
    	    ,country_code                => f_ADDRESS_rec.COUNTRY_NAME
          ,carrier_route               => f_CLN_ADDRESS_APP_rec.carrier_route
          ,delivery_point              => f_CLN_ADDRESS_APP_rec.delivery_point
          ,check_digit                 => f_CLN_ADDRESS_APP_rec.check_digit
          ,postnet_barcode             => f_CLN_ADDRESS_APP_rec.postnet_barcode
          ,last_line                   => f_CLN_ADDRESS_APP_rec.last_line
          ,suggestion_list             => f_CLN_ADDRESS_APP_rec.suggestion_list
          ,latitude                    => f_CLN_ADDRESS_APP_rec.latitude
          ,longitude                   => f_CLN_ADDRESS_APP_rec.longitude
    	    ,address_error_code          => f_CLN_ADDRESS_APP_rec.ADDRESS_ERROR_CODE
    	    ,address_verified_date       => f_CLN_ADDRESS_APP_rec.ADDRESS_VERIFIED_DATE
          ,address_error_text          => address_error_text
          ,fv_plsql_error              => fv_plsql_error
          ,fv_address_type             => fv_address_type
          ,fv_object_name              => fv_object_name
          );

END Verify_CLN_ADDRESSES_Record;
/******************************************************************************************
 *  Procedure Name  :   Batch_Verify_CLN_ADDRESSES
 *
 *  Scope           :   PUBLIC
 *
 *  Description     :   Batch Verify the addresses in the Application Address table (CLN_ADDRESSES)
 *                      * fn_max_verify
 *                           - maximum number of address to verify
 *                      * fb_update
 *                           - TRUE  = update the record and show results
 *                           - FALSE = don't update, only show results
 *                      * fb_only_unverified
 *                           - TRUE  = Only update unverified records
 *                           - FALSE = Update ALL records
 *                      * fv_address_type
 *                           - Application Address Type - only verify this type
 *                           - NULL = to verify all address types
 *                      * fn_days_back
 *                           - Verify Addresses that were updated/created this many days back
 *                             - Note: You can enter fractional days such as 0.25, 0.5, etc
 *                           - NULL = all addresses
 *                      * fb_skip_international
 *                           - Skip international address verification in batch mode
 *                      * fb_verify_inactive
 *                           - Verify inactive addresses as well as active ones
 *                           - This essentially ignores the "to date" and status indicator on the record
 *                      * fd_from_date / fd_to_date
 *                           - Specify a date range for the verification based on EFFDT
 *                           - This is useful for segmenting the data to create parallel batch procedures,
 *                             increasing throughput
 *                      * fv_state
 *                           - Specify an individual state to verify
 ******************************************************************************************/
PROCEDURE Batch_Verify_CLN_ADDRESSES (
                     fn_max_verify                  IN     NUMBER   DEFAULT 1000000
                    ,fb_update                      IN     BOOLEAN  DEFAULT TRUE
                    ,fb_only_unverified             IN     BOOLEAN  DEFAULT TRUE
                    ,fv_address_type                IN     VARCHAR2 DEFAULT NULL /* ADDRESS_TYPE */
                    ,fn_days_back                   IN     NUMBER   DEFAULT NULL
                    ,fb_skip_international          IN     BOOLEAN  DEFAULT TRUE
                    ,fb_verify_inactive             IN     BOOLEAN  DEFAULT FALSE
                    ,fd_from_date                   IN     DATE     DEFAULT NULL
                    ,fd_to_date                     IN     DATE     DEFAULT NULL
                    ,fv_state                       IN     VARCHAR2 DEFAULT NULL
                    )
IS
  ln_only_unverified    NUMBER := 0;
  ln_skip_international NUMBER := 0;
  ln_verify_inactive    NUMBER := 0;
  --
  -- Define the cursor to get the addresses from Application
  --
  cursor c_addr IS
    select a.rowid
          ,ca.rowid cln_rowid
          ,a.ADDRESS_ID       primary_key
          ,NVL(a.ADDRESS_TYPE_CODE, 'x') address_type
          ,a.ADDRESS_ID       ID_NUMBER
          ,a.UPDATED_DATE     EFFECTIVE_DATE
          ,company_name       company_name
          ,decode(a.ADDRESS_LINE_1, NULL, a.STREET_ADDRESS,  a.ADDRESS_LINE_1)  address_line_1
          ,decode(a.ADDRESS_LINE_1, NULL, a.SUITE_APARTMENT, a.ADDRESS_LINE_2)  address_line_2
          ,decode(a.ADDRESS_LINE_1, NULL, NULL,              a.ADDRESS_LINE_3)  address_line_3
          ,decode(a.ADDRESS_LINE_1, NULL, NULL,              a.ADDRESS_LINE_4)  address_line_4
          ,a.CITY             city
          ,a.STATE            state
          ,a.POSTAL_CODE      postal_code
          ,a.COUNTY_NAME      county_name
          ,a.COUNTRY_NAME     country_code
    from   CLN_ADDRESS_APP ca
          ,CLN_ADDRESSES a  /* Application Address Table */
    where  a.ADDRESS_ID          = ca.ID_NUMBER(+)
    and    NVL(a.ADDRESS_TYPE_CODE,'x')   = NVL(ca.ADDRESS_TYPE(+), 'x')
    and    a.UPDATED_DATE        = ca.EFFECTIVE_DATE(+)
    and    ca.FROM_TABLE(+) = 'CLN_ADDRESSES'
    -- Don't overwrite addresses where user checked "Override Address Verification"
    and    NVL(ca.ADDRESS_ERROR_CODE,'x') != '!'
    -- Only Local Country addresses if fb_Skip_International is set TRUE
    and    (  NVL(trim(a.COUNTRY_NAME),NVL(gv_local_country,'x')) = NVL(gv_local_country,'x')
           or ln_skip_international = 0
           )
    -- Only get addresses not verified if specified
    and    (  ln_only_unverified = 0
           or ca.ADDRESS_VERIFIED_DATE IS NULL
           )
    and    (  fv_address_type IS NULL
           or a.ADDRESS_TYPE_CODE = fv_address_type
           )
    and    (  fv_state IS NULL
           or a.state = fv_state
           )
    -- Check for from/to date parameters
    and    a.UPDATED_DATE between NVL(trunc(fd_from_date),a.UPDATED_DATE) and NVL(trunc(fd_to_date)+0.99999,a.UPDATED_DATE)
    and    (  fn_days_back IS NULL
           or NVL(a.UPDATED_DATE,sysdate) >= sysdate - fn_days_back
           );
  c_addr_rec c_addr%ROWTYPE;
  -- CLEAN_Address record structures
  l_address_rec       CLEAN_Address.Address_rec;
  l_address_rec_NULL  CLEAN_Address.Address_rec;
  -- Internal Statistic Variables
  li_start            NUMBER;
  li_end              NUMBER;
  li_count            NUMBER := 0;
  li_error_count      NUMBER := 0;
  lv_plsql_error      VARCHAR2(2000);
BEGIN
  dbms_output.enable(1000000);
  -- Set the number of address lines to process
  CLEAN_Address_APP.Set_Address_Lines(4);
  --
  -- Call the Pre Batch Address Verify User Exit
  --
  CLEAN_Address_APP_UE.Pre_Batch_Address_Verify;
  -- Clear the results table
  CLEAN_Address_APP.Delete_Results;
  -- Set the only unverified flag
  if fb_only_unverified then
    ln_only_unverified := 1;
  end if;
  -- Set the Skip International flag
  if fb_skip_international then
    ln_skip_international := 1;
    CLEAN_Address_APP.Set_Skip_International(TRUE);
  else
    CLEAN_Address_APP.Set_Skip_International(FALSE);
  end if;
  -- Set the Verify Inactive flag
  if fb_verify_inactive then
    ln_verify_inactive := 1;
  end if;
  --
  -- Start the timer
  --
  li_start := dbms_utility.get_time;
  --
  -- Loop through all addresses from the cursor
  --
  OPEN c_addr;
  LOOP
  BEGIN
    FETCH c_addr
    INTO  c_addr_rec;
    EXIT WHEN c_addr%NOTFOUND;
    --
    -- Initialize address record to NULL
    --
    l_address_rec := l_address_rec_NULL;
    --
    -- Assign the input variables to the address record
    --
    l_address_rec.Company        := c_addr_rec.company_name;
    l_address_rec.Address_Line_1 := c_addr_rec.address_line_1;
    l_address_rec.Address_Line_2 := c_addr_rec.address_line_2;
    l_address_rec.Address_Line_3 := c_addr_rec.address_line_3;
    l_address_rec.Address_Line_4 := c_addr_rec.address_line_4;
    l_address_rec.City           := c_addr_rec.City;
    l_address_rec.State          := c_addr_rec.State;
    l_address_rec.Postal_code    := c_addr_rec.postal_code;
    l_address_rec.county_name    := c_addr_rec.county_name;
    l_address_rec.country_code   := c_addr_rec.country_code;
    --
    -- Check for an address override to honor in batch mode
    -- ' .' at the end of address line 1 indicates an overridden address
    --
    if NVL(c_addr_rec.address_line_1,'x') NOT like '% .' then
      --
      -- Verify the address
      --
      CLEAN_Address_APP.Verify_CLEAN_Address_Record(
                    f_address_rec   => l_address_rec
                   ,fv_plsql_error  => lv_plsql_error
                   ,fv_address_type => c_addr_rec.ADDRESS_TYPE
                   ,fv_object_name  => 'Batch_Verify_CLN_ADDRESSES'
                   );
      -- Make sure there wasn't an Oracle Pipe error
      if lv_plsql_error IS NOT NULL then
        raise NO_DATA_FOUND;
      end if;
    else
      -- Set the override flag and remove the trailing ' .'
      l_address_rec.address_line_1 := substr(l_address_rec.address_line_1, 1, length(l_address_rec.address_line_1)-2);
      l_address_rec.error_code := '!';
      l_address_rec.error_string := 'Address Override';
    end if;
    --
    -- Add the error statistic to the output table
    --
    CLEAN_Address_APP.Add_Result(l_address_rec.error_code, l_address_rec.error_string);
    --
    -- See if we need to update the record
    --
    if fb_update then
      --
      -- Update the Application Address record
      --
      -- Only update existing if parameter is set, otherwise, inactivate and create a new address
      if gb_Batch_Update_Existing then
        --
        -- Update the existing record
        --
        update CLN_ADDRESSES set
            ADDRESS_LINE_1         = l_address_rec.address_line_1
           ,ADDRESS_LINE_2         = l_address_rec.address_line_2
           ,ADDRESS_LINE_3         = l_address_rec.address_line_3
           ,ADDRESS_LINE_4         = l_address_rec.address_line_4
           ,CITY                   = l_address_rec.city
           ,STATE                  = l_address_rec.state
           ,POSTAL_CODE            = l_address_rec.postal_code
           ,COUNTY_NAME            = l_address_rec.county_name
           ,COUNTRY_NAME           = l_address_rec.country_code
        where rowid = c_addr_rec.rowid;
        --
        -- Either insert or update the custom address table
        --
        if c_addr_rec.cln_rowid IS NOT NULL then
          -- Update the existing additional address record
          update CLN_ADDRESS_APP set
              CARRIER_ROUTE         = l_address_rec.carrier_route
             ,DELIVERY_POINT        = l_address_rec.DELIVERY_POINT
             ,CHECK_DIGIT           = l_address_rec.CHECK_DIGIT
             ,POSTNET_BARCODE       = l_address_rec.Address_Line_7
             ,LAST_LINE             = l_address_rec.LAST_LINE
             ,ADDRESS_VERIFIED_DATE = to_date(l_address_rec.Address_Line_8, 'YYYYMMDD')
             ,ADDRESS_ERROR_CODE    = l_address_rec.Error_Code
             ,SUGGESTION_LIST       = l_address_rec.error_string
             ,latitude              = l_address_rec.latitude
             ,longitude             = l_address_rec.longitude
          where rowid = c_addr_rec.cln_rowid;
        else
          -- Insert a new additional address record
          insert into CLN_ADDRESS_APP (
             ID_NUMBER
            ,FROM_TABLE
            ,ADDRESS_TYPE
            ,EFFECTIVE_DATE
            ,CARRIER_ROUTE
            ,DELIVERY_POINT
            ,CHECK_DIGIT
            ,POSTNET_BARCODE
            ,LAST_LINE
            ,ADDRESS_VERIFIED_DATE
            ,ADDRESS_ERROR_CODE
            ,SUGGESTION_LIST
            ,LATITUDE
            ,LONGITUDE
          ) values (
             c_addr_rec.ID_NUMBER
            ,'CLN_ADDRESSES'
            ,c_addr_rec.ADDRESS_TYPE
            ,c_addr_rec.EFFECTIVE_DATE
            ,l_address_rec.CARRIER_ROUTE
            ,l_address_rec.DELIVERY_POINT
            ,l_address_rec.CHECK_DIGIT
            ,l_address_rec.Address_Line_7
            ,l_address_rec.LAST_LINE
            ,to_date(l_address_rec.Address_Line_8, 'YYYYMMDD')
            ,l_address_rec.Error_Code
            ,l_address_rec.error_string
            ,l_address_rec.latitude
            ,l_address_rec.longitude
          );
        end if;
      end if;
      --
      -- Commit every 50 records
      --
      if mod(li_count+1, 50) = 0 then
        commit;
      end if;
    end if;
    --
    -- Increment the counter
    --
    li_count := li_count + 1;
    --
    -- Leave if we've reached the max counter
    --
    if li_count >= NVL(fn_max_verify, li_count+1) then
      EXIT;
    end if;
  EXCEPTION
    WHEN OTHERS THEN
      --
      -- fv_plsql_error was found - Oracle Pipe error
      --
      -- Show the error message and input/verified address
      dbms_output.put_line(chr(10)||'Unable to Verify Address:'
                ||chr(10)||'['||NVL(lv_plsql_error,SQLERRM)||']');
      dbms_output.put_line('*** Primary Key - Address Type: '
                         ||c_addr_rec.primary_key
                    ||'-'||c_addr_rec.ADDRESS_TYPE
                    );
      dbms_output.put_line('*** INPUT Address: '
                ||chr(10)||c_addr_rec.address_line_1
                ||ltrim(chr(10)||c_addr_rec.address_line_2
                ||chr(10)||c_addr_rec.address_line_3, chr(10))
                ||chr(10)||c_addr_rec.city||', '||c_addr_rec.state||' '||c_addr_rec.postal_code
                );
      dbms_output.put_line('*** VERIFIED Address: '
                ||chr(10)||l_address_rec.address_line_1
                ||ltrim(chr(10)||l_address_rec.address_line_2
                ||chr(10)||l_address_rec.address_line_3, chr(10))
                ||chr(10)||l_address_rec.city||', '||l_address_rec.state||' '||l_address_rec.postal_code
                );
      -- Increment error count
      li_error_count := li_error_count + 1;
      -- Leave if we get more than 50 errors
      if li_error_count > 50 then
        EXIT;
      end if;
  END;
  END LOOP;
  CLOSE c_addr;
  commit;
  --
  -- Get the end time
  --
  li_end := dbms_utility.get_time;
  --
  -- Display the Output
  --
  dbms_output.put_line(chr(10));
  -- Show Update Flag
  if fb_update then
    commit;
    dbms_output.put_line('Record Update : ON');
  else
    dbms_output.put_line('Record Update : OFF');
  end if;
  --
  -- Show Statistics
  --
  dbms_output.put_line('# Records  : '||to_char(li_count));
  dbms_output.put_line('# Errors   : '||to_char(li_error_count));
  dbms_output.put_line('Total Time : '||to_char((li_end - li_start)/100)||' sec');
  if (li_end - li_start) != 0 then
    dbms_output.put_line('# Rec/Hour : '||to_char(round(360000*li_count/(li_end - li_start))));
  end if;
  --
  -- Call the Post Batch Address Verify User Exit
  --
  CLEAN_Address_APP_UE.Post_Batch_Address_Verify;
  --
  -- Show the error summary
  --
  CLEAN_Address_APP.Show_Results;
  -- Delete the results table to clean up
  CLEAN_Address_APP.Delete_Results;
END Batch_Verify_CLN_ADDRESSES;
/******************************************************************************************
 *  Function Name   :   Get_Error_Text
 *
 *  Scope           :   PUBLIC
 *
 *  Description     :   Return the error text from the code in the ADDRESS_ERROR_CODE field
 *                      NOTE: This is a new column added to the table with the script:
 *                            pkg/address_table_mods.sql
 ******************************************************************************************/
FUNCTION Get_Error_Text (
                     address_error_code             IN     VARCHAR2 /* ADDRESS_ERROR_CODE */
                    ) RETURN VARCHAR2
IS
BEGIN
  if address_error_code IS NULL then
    return NULL;
  end if;
  if address_error_code like 'AD%' then
    -- Get international address error
    return cln$lookup.get_address_status(address_error_code);
  end if;
  return cln$lookup.get_address_error(address_error_code);
END Get_Error_Text;
/******************************************************************************************
 *  Function Name   :   Get_Error_Help
 *
 *  Scope           :   PUBLIC
 *
 *  Description     :   Return the error help text from the code in the ADDRESS_ERROR_CODE field
 *                      NOTE: This is a new column added to the table with the script:
 *                            pkg/address_table_mods.sql
 *                      NOTE: You can update the help text in the view CLN_Address_Errors_V
 *                            by updating the HELP_TEXT field, or you can override it by
 *                            using the COMMENT_TEXT field
 ******************************************************************************************/
FUNCTION Get_Error_Help (
                     address_error_code             IN     VARCHAR2 /* ADDRESS_ERROR_CODE */
                    ) RETURN VARCHAR2
IS
BEGIN
  if address_error_code IS NULL then
    return NULL;
  end if;
  return NVL(cln$lookup.Comment_Text(address_error_code, 'ADDRERROR')
            ,cln$lookup.Help_Text(address_error_code, 'ADDRERROR')
            );
END Get_Error_Help;
/******************************************************************************************
 *  Procedure Name  :   Check_Telephone
 *
 *  Scope           :   PUBLIC
 *
 *  Description     :   Verify a single telephone number for Application
 *
 *                      This call wraps the Telephone record implementation and excepts pure
 *                      VARCHAR2 IN/OUT parameters.
 *
 *                      telephone_results - <suggested_area>:<distance>:<error_code>
 *                         - Telephone suggested Area, Distance, and error code, separated by :
 ******************************************************************************************/
PROCEDURE Check_Telephone (
                     phone_area                     IN OUT VARCHAR2
                    ,phone_number                   IN OUT VARCHAR2
                    ,phone_extension                IN OUT VARCHAR2
                    ,postal_code                    IN     VARCHAR2
                    ,suggested_area                    OUT VARCHAR2 /* Possible new area code */
                    ,distance                          OUT NUMBER   /* Distance from Phone to ZIP Code */
                    ,telephone_error_text           IN OUT VARCHAR2
                    ,fv_plsql_error                 IN OUT VARCHAR2
                    ,fv_telephone_code              IN     VARCHAR2 DEFAULT NULL /* PHONE_TYPE */
                    ,fv_object_name                 IN     VARCHAR2 DEFAULT NULL /* Object Name where procedure is called from */
                    )
IS
  -- CLEAN_Address record structure
  l_telephone_rec       CLEAN_Address.Phone_rec;
  lv_area_code          VARCHAR2(10);
BEGIN
  -- Leave if the postal_code is blank - can't do anything
  if postal_code IS NULL
     or phone_number IS NULL then
    return;
  end if;
  --
  -- Backup existing area code
  --
  lv_area_code := phone_area;
  --
  -- 5/16/07 - Check for last 4 of phone number in extension field
  --
  if length(phone_number) = 3
     and length(phone_extension) = 4 then
    phone_number := phone_number||phone_extension;
    phone_extension := NULL;
  end if;
  --
  -- Assign the input variables to the telephone record
  --
  l_telephone_rec.Phone_Number := phone_area||phone_number;
  l_telephone_rec.Postal_Code  := postal_code;
  l_telephone_rec.Phone_Extension := phone_extension;
  l_telephone_rec.Phone_Number_Output_Format := '999-999-9999';
  telephone_error_text         := NULL;
  --
  -- Check the Telephone
  --
  Check_Telephone_Record(
                f_telephone_rec   => l_telephone_rec
               ,fv_phone_area     => phone_area
               ,fv_plsql_error    => fv_plsql_error
               ,fv_telephone_code => fv_telephone_code
               ,fv_object_name    => fv_object_name
               );
  phone_number := l_telephone_rec.Phone_Number;
  -- Get the suggested area code and distance
  suggested_area := l_telephone_rec.New_Phone_Area_Code;
  distance       := l_telephone_rec.Phone_Distance;
  -- Assign the area code back only if this was an area code SPLIT
  phone_area     := NVL(l_telephone_rec.Phone_Area_Code, phone_area);
  if l_telephone_rec.Phone_Check_Error_Code IS NOT NULL then
    telephone_error_text := cln$lookup.Get_Phone_Error(l_telephone_rec.Phone_Check_Error_Code);
  end if;
  --
  -- Assign the friendly error message based on results
  --
  if telephone_error_text IS NULL then
    --
    -- Check for Phone area code split
    --
    if phone_area != NVL(lv_area_code, phone_area) then
      telephone_error_text := 'Area Code updated due to area code split:'
               ||chr(10)||'OLD: '||lv_area_code
               ||chr(10)||'NEW: '||phone_area;
    --
    -- Check for a valid distance from the area code to the ZIP Code
    --
    elsif NVL(distance, -1) > NVL(gn_Telephone_Distance_Limit, 0)
          and NVL(distance, -1) != 9999 then
      telephone_error_text := 'WARNING: The Distance from the Area Code to the ZIP Code ('
               ||to_char(distance)
               ||' miles) exceeds the threshhold of '||to_char(gn_Telephone_Distance_Limit)||' miles!';
      -- conditionally show the Suggested Area Code
      if suggested_area IS NOT NULL then
        telephone_error_text := telephone_error_text||chr(10)||'Suggested Area Code: '||suggested_area;
      end if;
    --
    -- Show any "suggested" area codes if the input area code is not valid
    --
    elsif suggested_area IS NOT NULL
          and suggested_area != NVL(phone_area,'x') then
      telephone_error_text := 'WARNING: A corrected Area Code is suggested based on the zip code entered.'
               ||chr(10)||'Suggested Area Code: '||suggested_area;
    end if;
  end if;
END Check_Telephone;
/******************************************************************************************
 *  Procedure Name  :   Check_PHONE_Record
 *
 *  Scope           :   PRIVATE
 *
 *  Description     :   Verify a single telephone by passing in the CLN_ADDRESSES Record type
 ******************************************************************************************/
PROCEDURE Check_PHONE_Record (
                     f_PHONE_rec                    IN OUT CLN_ADDRESSES%ROWTYPE
                    ,fv_postal                      IN     VARCHAR2
                    ,suggested_area                    OUT VARCHAR2 /* Possible new area code */
                    ,distance                          OUT NUMBER   /* Distance from Phone to ZIP Code */
                    ,telephone_error_text           IN OUT VARCHAR2
                    ,fv_plsql_error                 IN OUT VARCHAR2
                    ,fv_telephone_code              IN     VARCHAR2 DEFAULT NULL /* PHONE_TYPE */
                    ,fv_object_name                 IN     VARCHAR2 DEFAULT NULL /* Object Name where procedure is called from */
                    )
IS
  lv_area VARCHAR2(10);
BEGIN
  --
  -- Call the CLEAN_Address Application Verify Telephone procedure with individual parameters
  --
  CLEAN_Address_APP.Check_Telephone (
           phone_area           => lv_area
          ,phone_number         => f_PHONE_rec.TELEPHONE_1
          ,phone_extension      => f_PHONE_rec.TELEPHONE_1_EXTENSION
          ,postal_code          => fv_postal
          ,suggested_area       => suggested_area
          ,distance             => distance
          ,telephone_error_text => telephone_error_text
          ,fv_plsql_error       => fv_plsql_error
          ,fv_telephone_code    => fv_telephone_code
          ,fv_object_name       => fv_object_name
          );
END Check_PHONE_Record;
/******************************************************************************************
 *  Procedure Name  :   Batch_Check_Telephone
 *
 *  Scope           :   PUBLIC
 *
 *  Description     :   Batch Verify the telephones in the Application Telephone table (CLN_ADDRESSES)
 *                      * fn_max_verify
 *                           - maximum number of telephones to verify
 *                      * fb_update
 *                           - TRUE  = update the record and show results
 *                           - FALSE = don't update, only show results
 *                      * fb_only_unverified
 *                           - TRUE  = Only update unverified records
 *                           - FALSE = Update ALL records
 *                      * fv_telephone_code
 *                           - Application Telephone Code - only verify this type (SPRTELE_TELE_CODE)
 *                           - NULL = to verify all telephone codes
 *                      * fn_days_back
 *                           - Verify Telephone numbers that were updated/created this many days back
 *                             - Note: You can enter fractional days such as 0.25, 0.5, etc
 *                           - NULL = all telephones
 ******************************************************************************************/
PROCEDURE Batch_Check_Telephone (
                     fn_max_verify                  IN     NUMBER   DEFAULT 1000000
                    ,fb_update                      IN     BOOLEAN  DEFAULT TRUE
                    ,fb_only_unverified             IN     BOOLEAN  DEFAULT TRUE
                    ,fv_telephone_code              IN     VARCHAR2 DEFAULT NULL /* SPRTELE_TELE_CODE */
                    ,fn_days_back                   IN     NUMBER   DEFAULT NULL
                    )
IS
  ln_only_unverified   NUMBER := 0;
  --
  -- Define the cursor to get the addresses from Application
  --
  cursor c_tele IS
    select t.rowid
          ,t.ADDRESS_ID
          ,t.TELEPHONE_1_TYPE_CODE
          ,NULL                  phone_area
          ,TELEPHONE_1           phone_number
          ,TELEPHONE_1_EXTENSION phone_ext
          ,NULL      postal_code
          ,NULL      country_code
    from   CLN_ADDRESSES t
    where  1=1  --FIX LOGIC NVL(ltrim(upper(SPRTELE_STATUS_IND)),'A') = 'A'
    and    t.TELEPHONE_1 IS NOT NULL
    and    (  fv_telephone_code IS NULL
           or t.TELEPHONE_1_TYPE_CODE = fv_telephone_code
           )
    and    (  fn_days_back IS NULL
           or t.UPDATED_DATE > sysdate - fn_days_back
           );
  c_tele_rec c_tele%ROWTYPE;
  -- CLEAN_Address record structures
  l_telephone_rec       CLEAN_Address.Phone_rec;
  l_telephone_rec_NULL  CLEAN_Address.Phone_rec;
  -- Internal Statistic Variables
  li_start            NUMBER;
  li_end              NUMBER;
  li_count            NUMBER := 0;
  li_error_count      NUMBER := 0;
  lv_plsql_error      VARCHAR2(2000);
BEGIN
  dbms_output.enable(1000000);
  --
  -- Call the Pre Batch Telephone Check User Exit
  --
  CLEAN_Address_APP_UE.Pre_Batch_Telephone_Check;
  -- Clear the results table
  Delete_Results;
  -- Set the only unverified flag
  if fb_only_unverified then
    ln_only_unverified := 1;
  end if;
  --
  -- Start the timer
  --
  li_start := dbms_utility.get_time;
  --
  -- Loop through all addresses from the cursor
  --
  OPEN c_tele;
  LOOP
  BEGIN
    FETCH c_tele
    INTO  c_tele_rec;
    EXIT WHEN c_tele%NOTFOUND;
    --
    -- Initialize address record to NULL
    --
    l_telephone_rec := l_telephone_rec_NULL;
    --
    -- 5/16/07 - Check for last 4 of phone number in extension field
    --
    if length(c_tele_rec.phone_number) = 3
       and length(c_tele_rec.phone_ext) = 4 then
      c_tele_rec.phone_number := c_tele_rec.phone_number||c_tele_rec.phone_ext;
      c_tele_rec.phone_ext := NULL;
    end if;
    --
    -- Assign the input variables to the address record
    --
    l_telephone_rec.Phone_Number := c_tele_rec.phone_number;
    l_telephone_rec.Phone_Extension := c_tele_rec.phone_ext;
    l_telephone_rec.Postal_Code  := c_tele_rec.postal_code;
    l_telephone_rec.Phone_Number_Output_Format := '9999999999';
    --
    -- Verify the Address
    --
    Check_Telephone_Record(
                  f_telephone_rec   => l_telephone_rec
                 ,fv_phone_area     => c_tele_rec.phone_area
                 ,fv_plsql_error    => lv_plsql_error
                 ,fv_telephone_code => c_tele_rec.TELEPHONE_1_TYPE_CODE
                 ,fv_object_name    => 'Batch_Check_Telephone'
                 );
    -- Make sure there wasn't an Oracle Pipe error
    if lv_plsql_error IS NOT NULL then
      raise NO_DATA_FOUND;
    end if;
    --
    -- Add the error statistic to the output table
    --
    Add_Result(l_telephone_rec.Phone_Check_Error_Code
              ,cln$lookup.Get_Phone_Error(l_telephone_rec.Phone_Check_Error_Code)
              );
    --
    -- See if we need to update the record
    --
    if fb_update then
      --
      -- Update the Application Address record
      --
      update CLN_ADDRESSES set
          TELEPHONE_1           = l_telephone_rec.phone_number
         ,TELEPHONE_1_EXTENSION = l_telephone_rec.phone_extension
      where rowid = c_tele_rec.rowid;
      --
      -- Commit every 50 records
      --
      if mod(li_count+1, 50) = 0 then
        commit;
      end if;
    end if;
    --
    -- Increment the counter
    --
    li_count := li_count + 1;
    --
    -- Leave if we've reached the max counter
    --
    if li_count >= NVL(fn_max_verify, li_count+1) then
      EXIT;
    end if;
  EXCEPTION
    WHEN OTHERS THEN
      lv_plsql_error := NVL(lv_plsql_error, SQLERRM);
      --
      -- fv_plsql_error was found - Oracle Pipe error
      --
      -- Show the error message and input/verified address
      dbms_output.put_line(chr(10)||'Unable to Verify Telephone:'
                ||chr(10)||lv_plsql_error);
      dbms_output.put_line('*** Primary Key: '
                         ||to_char(c_tele_rec.ADDRESS_ID)
                    ||'-'||c_tele_rec.TELEPHONE_1_TYPE_CODE
                    );
      dbms_output.put_line('*** INPUT Telephone: '
                ||c_tele_rec.phone_area||' '||c_tele_rec.phone_number
                );
      -- Increment error count
      li_error_count := li_error_count + 1;
      -- Leave if we get more than 50 errors
      if li_error_count > 50 then
        EXIT;
      end if;
  END;
  END LOOP;
  CLOSE c_tele;
  commit;
  --
  -- Get the end time
  --
  li_end := dbms_utility.get_time;
  --
  -- Display the Output
  --
  dbms_output.put_line(chr(10));
  -- Show Update Flag
  if fb_update then
    commit;
    dbms_output.put_line('Record Update : ON');
  else
    dbms_output.put_line('Record Update : OFF');
  end if;
  --
  -- Show Statistics
  --
  dbms_output.put_line('# Records  : '||to_char(li_count));
  dbms_output.put_line('# Errors   : '||to_char(li_error_count));
  dbms_output.put_line('Total Time : '||to_char((li_end - li_start)/100)||' sec');
  if (li_end - li_start) != 0 then
    dbms_output.put_line('# Rec/Hour : '||to_char(round(360000*li_count/(li_end - li_start))));
  end if;
  --
  -- Call the Post Batch Telephone Check User Exit
  --
  CLEAN_Address_APP_UE.Post_Batch_Telephone_Check;
  --
  -- Show the error summary
  --
  Show_Results;
  -- Delete the results table to clean up
  Delete_Results;
END Batch_Check_Telephone;
/******************************************************************************************
 *  Function Name   :   Get_Telephone_Error_Text
 *
 *  Scope           :   PUBLIC
 *
 *  Description     :   Return the error text from the code in the SPRTELE_DATA_ORIGIN field
 *                      NOTE: The SPRTELE_DATA_ORIGIN holds the telephone results in the format
 *                            <suggested_area>:<distance>:<error_code>
 ******************************************************************************************/
FUNCTION Get_Telephone_Error_Text (
                     fv_error_code                  IN     VARCHAR2
                    ) RETURN VARCHAR2
IS
BEGIN
  if fv_error_code IS NULL then
    return NULL;
  end if;
  return cln$lookup.Get_Phone_Error(fv_error_code);
END Get_Telephone_Error_Text;
/******************************************************************************************
 *  Procedure Name  :   Standardize_Name
 *
 *  Scope           :   PUBLIC
 *
 *  Description     :   Standardize the name components and return warning message if first name appears invalid
 *
 *                      This call wraps the Name record implementation and excepts pure
 *                      VARCHAR2 IN/OUT parameters.
 ******************************************************************************************/
PROCEDURE Standardize_Name (
                     first_name                     IN OUT VARCHAR2 /* SPRIDEN_FIRST_NAME */
                    ,last_name                      IN OUT VARCHAR2 /* SPRIDEN_LAST_NAME */
                    ,middle_name                    IN OUT VARCHAR2 /* SPRIDEN_MI */
                    ,prefix                         IN OUT VARCHAR2 /* SPBPERS_NAME_PREFIX */
                    ,suffix                         IN OUT VARCHAR2 /* SPBPERS_NAME_SUFFIX */
                    ,name_error_code                IN OUT VARCHAR2
                    ,name_error_text                IN OUT VARCHAR2
                    ,fv_plsql_error                 IN OUT VARCHAR2
                    ,fv_name_type_code              IN     VARCHAR2 DEFAULT NULL /* SPRIDEN_NTYP_CODE */
                    ,fv_object_name                 IN     VARCHAR2 DEFAULT NULL /* Object Name where procedure is called from */
                    )
IS
  l_name_rec        CLEAN_Address.NAME_REC;
BEGIN
  --
  -- Assign the input variables to the name record
  --
  l_name_rec.first_name      := first_name;
  l_name_rec.last_name       := last_name;
  l_name_rec.middle_name     := middle_name;
  l_name_rec.name_prefix     := prefix;
  l_name_rec.name_suffix     := suffix;
  name_error_text            := NULL;
  --
  -- Standardize the Name
  --
  Standardize_Name_Record (
           f_name_rec          => l_name_rec
          ,fv_name_error_text  => name_error_text
          ,fv_plsql_error      => fv_plsql_error
          ,fv_name_type_code   => fv_name_type_code
          ,fv_object_name      => fv_object_name
          );
  --
  -- Remap the output name fields
  --
  first_name      := l_name_rec.first_name;
  last_name       := l_name_rec.last_name;
  middle_name     := l_name_rec.middle_name;
  prefix          := l_name_rec.name_prefix;
  suffix          := l_name_rec.name_suffix;
  name_error_code := l_name_rec.name_parse_status_code;
END Standardize_Name;
/******************************************************************************************
 *  Function Name   :   Get_Name_Error_Text
 *
 *  Scope           :   PUBLIC
 *
 *  Description     :   Return the error text from the name error code
 ******************************************************************************************/
FUNCTION Get_Name_Error_Text (
                     name_error_code                IN     VARCHAR2
                    ) RETURN VARCHAR2
IS
BEGIN
  if name_error_code IS NULL then
    return NULL;
  end if;
  if name_error_code = 'U' then
    return 'Unknown First Name';
  end if;
  return cln$lookup.Get_Name_Status(name_error_code);
END Get_Name_Error_Text;
/******************************************************************************************
 *  Procedure Name  :   Validate_Email
 *
 *  Scope           :   PUBLIC
 *
 *  Description     :   Validate the Email Address, checking syntax and domain name
 *
 *                      This call wraps the Email record implementation and excepts pure
 *                      VARCHAR2 IN/OUT parameters.
 ******************************************************************************************/
PROCEDURE Validate_Email (
                     email_address                  IN OUT VARCHAR2 /* GOREMAIL_EMAIL_ADDRESS */
                    ,email_error_code               IN OUT VARCHAR2
                    ,email_error_text               IN OUT VARCHAR2
                    ,fv_plsql_error                 IN OUT VARCHAR2
                    ,fv_email_type_code             IN     VARCHAR2 DEFAULT NULL /* GOREMAL_EMAL_CODE */
                    ,fv_object_name                 IN     VARCHAR2 DEFAULT NULL /* Object Name where procedure is called from */
                    )
IS
  l_email_rec       CLEAN_Address.EMAIL_REC;
BEGIN
  --
  -- Assign the input variables to the email record
  --
  l_email_rec.email_address := email_address;
  email_error_text          := NULL;
  --
  -- Validate the Email
  --
  Validate_Email_Record (
           f_email_rec         => l_email_rec
          ,fv_email_error_text => email_error_text
          ,fv_plsql_error      => fv_plsql_error
          ,fv_email_type_code  => fv_email_type_code
          ,fv_object_name      => fv_object_name
          );
  --
  -- Remap the output email fields
  --
  email_address := l_email_rec.email_address;
  email_error_code := l_email_rec.email_status_code;
END Validate_Email;
/******************************************************************************************
 *  Function Name   :   Get_Email_Error_Text
 *
 *  Scope           :   PUBLIC
 *
 *  Description     :   Return the error text from the email error code
 ******************************************************************************************/
FUNCTION Get_Email_Error_Text (
                     email_error_code               IN     VARCHAR2
                    ) RETURN VARCHAR2
IS
BEGIN
  if email_error_code IS NULL then
    return NULL;
  end if;
  return cln$lookup.Get_Email_Status(email_error_code);
END Get_Email_Error_Text;
-----------------------------------------------------------------------------------------
--
-- PUBLIC PROCEDURES - PARAMETER GET / SET
--
-----------------------------------------------------------------------------------------
/******************************************************************************************
 *  Function Name   :   Get_Expand_Street_Suffix
 *
 *  Description     :
 *    Set gb_expand_street_suffix to TRUE to expand the street suffix (Ave -> Avenue) if it will
 *    fit on one line.  If set to FALSE, the USPS CASS standardization rules will be used.
 *    DEFAULT: FALSE
 ******************************************************************************************/
Function Get_Expand_Street_Suffix RETURN BOOLEAN
IS
BEGIN
  return NVL(gb_Expand_Street_Suffix, FALSE);
END Get_Expand_Street_Suffix;
/******************************************************************************************
 *  Procedure Name  :   Set_Expand_Street_Suffix
 ******************************************************************************************/
Procedure Set_Expand_Street_Suffix (
                     fb_value                       IN     BOOLEAN  DEFAULT FALSE
                    )
IS
BEGIN
  gb_Expand_Street_Suffix := NVL(fb_value, FALSE);
END Set_Expand_Street_Suffix;
/******************************************************************************************
 *  Function Name   :   Get_Suite_Apt_Before_Street
 *
 *  Description     :
 *    Set gb_suite_apt_before_street to TRUE to put the suite/apartment line before the address line
 *    in cases where the the suite/apartment does not fit on one line.  If set to FALSE, the suite/apt
 *    will appear on the line after the street address line.
 *    DEFAULT: FALSE
 ******************************************************************************************/
Function Get_Suite_Apt_Before_Street RETURN BOOLEAN
IS
BEGIN
  return NVL(gb_Suite_Apt_Before_Street, FALSE);
END Get_Suite_Apt_Before_Street;
/******************************************************************************************
 *  Procedure Name  :   Set_Suite_Apt_Before_Street
 ******************************************************************************************/
Procedure Set_Suite_Apt_Before_Street (
                     fb_value                       IN     BOOLEAN  DEFAULT FALSE
                    )
IS
BEGIN
  gb_Suite_Apt_Before_Street := NVL(fb_value, FALSE);
END Set_Suite_Apt_Before_Street;
/******************************************************************************************
 *  Function Name   :   Get_Append_Suite_To_Street
 *
 *  Description     :
 *    Set gb_append_suite_to_street to TRUE to always "try" to append the suite/apartment line
 *    on the same line as the the address line - only when it will fit.  If set to FALSE, the
 *    suite/apt will be on a separate line if there a free line available
 *    DEFAULT: TRUE
 ******************************************************************************************/
Function Get_Append_Suite_To_Street RETURN BOOLEAN
IS
BEGIN
  return NVL(gb_Append_Suite_To_Street, TRUE);
END Get_Append_Suite_To_Street;
/******************************************************************************************
 *  Procedure Name  :   Set_Append_Suite_To_Street
 ******************************************************************************************/
Procedure Set_Append_Suite_To_Street (
                     fb_value                       IN     BOOLEAN  DEFAULT TRUE
                    )
IS
BEGIN
  gb_Append_Suite_To_Street := NVL(fb_value, TRUE);
END Set_Append_Suite_To_Street;
/******************************************************************************************
 *  Function Name   :   Get_Use_ZIP_Plus4
 *
 *  Description     :
 *    Set gb_use_ZIP_Plus4 to TRUE to append the ZIP+4 digits after the ZIP Code in the
 *    ZIP field.  If set to FALSE, only the 5-digit ZIP Code will be used
 *    DEFAULT: TRUE
 ******************************************************************************************/
Function Get_Use_ZIP_Plus4 RETURN BOOLEAN
IS
BEGIN
  return NVL(gb_Use_ZIP_Plus4, TRUE);
END Get_Use_ZIP_Plus4;
/******************************************************************************************
 *  Procedure Name  :   Set_Use_ZIP_Plus4
 ******************************************************************************************/
Procedure Set_Use_ZIP_Plus4 (
                     fb_value                       IN     BOOLEAN  DEFAULT TRUE
                    )
IS
BEGIN
  gb_Use_ZIP_Plus4 := NVL(fb_value, TRUE);
END Set_Use_ZIP_Plus4;
/******************************************************************************************
 *  Function Name   :   Get_Batch_Update_Existing
 *
 *  Description     :
 *    Set gb_batch_update_existing to TRUE to update the existing address record when running
 *    in batch mode.  If set to FALSE, a new record will be created if anything changes in the
 *    address.  This could easily double the size of the Address table if set FALSE, since most
 *    addresses are not keyed in in USPS standards. Please take this into consideration.
 *    DEFAULT: TRUE
 ******************************************************************************************/
Function Get_Batch_Update_Existing RETURN BOOLEAN
IS
BEGIN
  return NVL(gb_Batch_Update_Existing, TRUE);
END Get_Batch_Update_Existing;
/******************************************************************************************
 *  Procedure Name  :   Set_Batch_Update_Existing
 ******************************************************************************************/
Procedure Set_Batch_Update_Existing (
                     fb_value                       IN     BOOLEAN  DEFAULT TRUE
                    )
IS
BEGIN
  gb_Batch_Update_Existing := NVL(fb_value, TRUE);
END Set_Batch_Update_Existing;
/******************************************************************************************
 *  Function Name   :   Get_Telephone_Distance_Limit
 *
 *  Description     :
 *    Set gn_telephone_distance_limit to the number of miles threshhold to allow before an
 *    alert message is displayed to the user.  This is the distance between the telephone
 *    wire center (area code and prefix = first 6 digits) and the 5-digit ZIP Code
 *    DEFAULT: 100
 ******************************************************************************************/
Function Get_Telephone_Distance_Limit RETURN NUMBER
IS
BEGIN
  return NVL(gn_Telephone_Distance_Limit, 100);
END Get_Telephone_Distance_Limit;
/******************************************************************************************
 *  Procedure Name  :   Set_Telephone_Distance_Limit
 ******************************************************************************************/
Procedure Set_Telephone_Distance_Limit (
                     fn_value                       IN     NUMBER  DEFAULT 100
                    )
IS
BEGIN
  gn_Telephone_Distance_Limit := NVL(fn_value, 100);
END Set_Telephone_Distance_Limit;
/******************************************************************************************
 *  Function Name   :   Get_Error_Suggest_Count
 *
 *  Description     :
 *    Set gn_Error_Suggest_Count to the number of possible matches to return for unverified addresses
 *    DEFAULT: 0
 ******************************************************************************************/
Function Get_Error_Suggest_Count RETURN PLS_INTEGER
IS
BEGIN
  return NVL(gn_Error_Suggest_Count, 0);
END Get_Error_Suggest_Count;
/******************************************************************************************
 *  Procedure Name  :   Set_Error_Suggest_Count
 ******************************************************************************************/
Procedure Set_Error_Suggest_Count (
                     fn_value                       IN     PLS_INTEGER  DEFAULT 0
                    )
IS
BEGIN
  gn_Error_Suggest_Count := NVL(fn_value, 0);
END Set_Error_Suggest_Count;
/******************************************************************************************
 *  Function Name   :   Get_Upper_Case_Results
 *
 *  Description     :
 *    Set gb_Upper_Case_Results to TRUE to always UPPER CASE address fields
 *    DEFAULT: FALSE
 ******************************************************************************************/
Function Get_Upper_Case_Results RETURN BOOLEAN
IS
BEGIN
  return NVL(gb_Upper_Case_Results, FALSE);
END Get_Upper_Case_Results;
/******************************************************************************************
 *  Procedure Name  :   Set_Upper_Case_Results
 ******************************************************************************************/
Procedure Set_Upper_Case_Results (
                     fb_value                       IN     BOOLEAN  DEFAULT FALSE
                    )
IS
BEGIN
  gb_Upper_Case_Results := NVL(fb_value, FALSE);
END Set_Upper_Case_Results;
/******************************************************************************************
 *  Function Name   :   Get_Address_Lines
 *
 *  Description     :
 *    Set gn_Address_Lines to the number of address lines to process
 *    DEFAULT: 4
 ******************************************************************************************/
Function Get_Address_Lines RETURN NUMBER
IS
BEGIN
  return NVL(gn_Address_Lines, 4);
END Get_Address_Lines;
/******************************************************************************************
 *  Procedure Name  :   Set_Address_Lines
 ******************************************************************************************/
Procedure Set_Address_Lines (
                     fn_value                       IN     NUMBER  DEFAULT 4
                    )
IS
BEGIN
  gn_Address_Lines := NVL(fn_value, 4);
END Set_Address_Lines;
/******************************************************************************************
 *  Function Name   :   Get_Max_Address_Length
 *
 *  Description     :
 *    Set gn_Max_Address_Length to limit size of address field
 *    DEFAULT: 55
 ******************************************************************************************/
Function Get_Max_Address_Length RETURN NUMBER
IS
BEGIN
  return NVL(gn_Max_Address_Length, 55);
END Get_Max_Address_Length;
/******************************************************************************************
 *  Procedure Name  :   Set_Max_Address_Length
 ******************************************************************************************/
Procedure Set_Max_Address_Length (
                     fn_value                       IN     NUMBER  DEFAULT 55
                    )
IS
BEGIN
  gn_Max_Address_Length := NVL(fn_value, 55);
END Set_Max_Address_Length;
/******************************************************************************************
 *  Function Name   :   Get_Max_City_Length
 *
 *  Description     :
 *    Set gn_Max_City_Length to limit size of City field
 *    DEFAULT: 20
 ******************************************************************************************/
Function Get_Max_City_Length RETURN NUMBER
IS
BEGIN
  return NVL(gn_Max_City_Length, 20);
END Get_Max_City_Length;
/******************************************************************************************
 *  Procedure Name  :   Set_Max_City_Length
 ******************************************************************************************/
Procedure Set_Max_City_Length (
                     fn_value                       IN     NUMBER  DEFAULT 55
                    )
IS
BEGIN
  gn_Max_City_Length := NVL(fn_value, 20);
END Set_Max_City_Length;
/******************************************************************************************
 *  Function Name   :   Get_Max_State_Length
 *
 *  Description     :
 *    Set gn_Max_State_Length to limit size of State field
 *    DEFAULT: 6
 ******************************************************************************************/
Function Get_Max_State_Length RETURN NUMBER
IS
BEGIN
  return NVL(gn_Max_State_Length, 6);
END Get_Max_State_Length;
/******************************************************************************************
 *  Procedure Name  :   Set_Max_State_Length
 ******************************************************************************************/
Procedure Set_Max_State_Length (
                     fn_value                       IN     NUMBER  DEFAULT 55
                    )
IS
BEGIN
  gn_Max_State_Length := NVL(fn_value, 20);
END Set_Max_State_Length;
/******************************************************************************************
 *  Function Name   :   Get_Ignore_Error
 ******************************************************************************************/
Function Get_Ignore_Error RETURN BOOLEAN
IS
BEGIN
  return gb_Ignore_Error;
END Get_Ignore_Error;
/******************************************************************************************
 *  Procedure Name  :   Set_Ignore_Error
 *
 *  Description     :
 *    Set gb_Ignore_Error to TRUE to allow Web Self-Service to save data even with an error code
 *    - NOTE: The error code will still be stored on the record but allow the transaction to continue
 ******************************************************************************************/
Procedure Set_Ignore_Error (
                     fb_value                       IN     BOOLEAN
                    )
IS
BEGIN
  gb_Ignore_Error := fb_value;
END Set_Ignore_Error;
/******************************************************************************************
 *  Function Name   :   Get_Object_Name
 ******************************************************************************************/
Function Get_Object_Name RETURN VARCHAR2
IS
BEGIN
  return gv_Object_Name;
END Get_Object_Name;
/******************************************************************************************
 *  Procedure Name  :   Set_Object_Name
 *
 *  Description     :
 *    Set gv_Object_Name to the Object that is currently calling the verification procedure
 ******************************************************************************************/
Procedure Set_Object_Name (
                     fv_value                       IN     VARCHAR2
                    )
IS
BEGIN
  gv_Object_Name := fv_value;
END Set_Object_Name;
/******************************************************************************************
 *  Function Name   :   Get_Local_Country
 ******************************************************************************************/
Function Get_Local_Country RETURN VARCHAR2
IS
BEGIN
  return gv_Local_Country;
END Get_Local_Country;
/******************************************************************************************
 *  Procedure Name  :   Set_Local_Country
 *
 *  Description     :
 *    Set gv_Local_Country to the value specified, which is the Application specific
 *    Nation Code for the predominant country
 ******************************************************************************************/
Procedure Set_Local_Country (
                     fv_value                       IN     VARCHAR2
                    )
IS
BEGIN
  gv_Local_Country := fv_value;
END Set_Local_Country;
/******************************************************************************************
 *  Function Name   :   Get_Skip_International
 ******************************************************************************************/
Function Get_Skip_International RETURN BOOLEAN
IS
BEGIN
  return gb_Skip_International;
END Get_Skip_International;
/******************************************************************************************
 *  Procedure Name  :   Set_Skip_International
 *
 *  Description     :
 *    Set gb_Skip_International TRUE to not process international address records
 *    This allows more control on the purchased Web Service transactions.
 *    NOTE: User should pass this parameter into the batch procedure rather than setting directly
 ******************************************************************************************/
Procedure Set_Skip_International (
                     fb_value                       IN     BOOLEAN
                    )
IS
BEGIN
  gb_Skip_International := fb_value;
END Set_Skip_International;
/******************************************************************************************
 *  Function Name   :   Get_Current_URL
 *
 *  Description     :
 *    Return the current host, port, and PL/SQL DAD URL prefix - used in AJAX implementation
 ******************************************************************************************/
Function Get_Current_URL RETURN VARCHAR2
IS
BEGIN
  if NVL(owa_util.get_cgi_env('SERVER_PORT'),'80') not in ('80','443') then
    if instr(owa_util.get_cgi_env('HTTP_HOST'), owa_util.get_cgi_env('SERVER_PORT')) > 0 then
      return lower(owa_util.get_cgi_env('REQUEST_PROTOCOL'))||'://'
                 ||owa_util.get_cgi_env('HTTP_HOST')
                 ||owa_util.get_cgi_env('SCRIPT_NAME')||'/';
    else
      return lower(owa_util.get_cgi_env('REQUEST_PROTOCOL'))||'://'
                 ||owa_util.get_cgi_env('HTTP_HOST')
                 ||':'||owa_util.get_cgi_env('SERVER_PORT')
                 ||owa_util.get_cgi_env('SCRIPT_NAME')||'/';
    end if;
  end if;
  return lower(owa_util.get_cgi_env('REQUEST_PROTOCOL'))||'://'
             ||owa_util.get_cgi_env('HTTP_HOST')
             ||owa_util.get_cgi_env('SCRIPT_NAME')||'/';
END Get_Current_URL;
END CLEAN_Address_APP;

/

set scan on
