set scan off
Create or Replace Package Body CLEAN_Address_Wrapper
IS
gv_Package_Body_Version  CONSTANT VARCHAR2(30) := '1.0.1.0';
gv_Package_Body_Date     CONSTANT VARCHAR2(30) := '09-SEP-2004';
/*****************************************************************************************
 *
 *  Package Name    :   CLEAN_Address_Wrapper
 *
 *  Description     :   Wrapper package for CLEAN_Address for isolating specific fields
 *                      to pass to the verification request
 *
 *                      This package is meant to be copied or modified for your specific
 *                      implementation
 *
 *  NOTE: This is very useful for Java implementations since Java can map easier to
 *        primitive datatypes rather than PL/SQL record structures
 *
 *****************************************************************************************
 * Copyright (c) 2004 Runner Technologies, Inc.  All Rights Reserved.
 * www.RunnerTechnologies.com   (877)784-0003  561-395-9322
 *****************************************************************************************/
-----------------------------------------------------------------------------------------
-- PUBLIC PROCEDURES
-----------------------------------------------------------------------------------------
/******************************************************************************************
 *  Procedure Name  :   Verify
 *
 *  Scope           :   PUBLIC
 *
 *  Description     :   Verify an address
 *                      This call wraps the Address record implementation and excepts pure
 *                      VARCHAR2 IN/OUT parameters.  This is suited for calling from Java
 ******************************************************************************************/
PROCEDURE Verify (
                     address_name                   IN OUT VARCHAR2
                    ,address_1                      IN OUT VARCHAR2
                    ,address_2                      IN OUT VARCHAR2
                    ,city                           IN OUT VARCHAR2
                    ,state_province                 IN OUT VARCHAR2
                    ,postal_code                    IN OUT VARCHAR2
                    ,country_code                   IN OUT VARCHAR2
                    ,telephone_number               IN OUT VARCHAR2
                    ,telephone_format               IN     VARCHAR2
                    ,address_status_code               OUT VARCHAR2
                    ,address_error_code                OUT VARCHAR2
                    ,dpv_status_code                   OUT VARCHAR2
                    ,formatted_address_1               OUT VARCHAR2
                    ,formatted_address_2               OUT VARCHAR2
                    ,formatted_address_3               OUT VARCHAR2
                    ,formatted_address_4               OUT VARCHAR2
                    ,gender                            OUT VARCHAR2
                    ,telephone_distance                OUT NUMBER
                    ,latitude                          OUT NUMBER
                    ,longitude                         OUT NUMBER
                    ,fv_plsql_return_code              OUT VARCHAR2
                    ,fv_plsql_return_description       OUT VARCHAR2
                    )
IS
  l_address_rec      CLEAN_Address.ADDRESS_REC;
BEGIN
  ---------------------------------------------------------
  -- Assign the input variables to the Address record
  ---------------------------------------------------------
  l_address_rec.Address_Name   := address_name;
  l_address_rec.Address_Line_1 := address_1;
  l_address_rec.Address_Line_2 := address_2;
  l_address_rec.City           := city;
  l_address_rec.Province       := state_province;
  l_address_rec.Postal_Code    := postal_code;
  l_address_rec.Country_Code   := country_code;
  l_address_rec.Phone_Number   := telephone_number;
  l_address_rec.Phone_Number_Output_Format := telephone_format;
  ---------------------------------------------------------
  -- Verify the address
  ---------------------------------------------------------
  CLEAN_Address.Verify_Generic_Silent(
           f_address_rec              => l_address_rec
          ,fn_number_address_lines    => 2
          ,fv_return_code             => fv_plsql_return_code
          ,fv_return_description      => fv_plsql_return_description
          );
  ---------------------------------------------------------
  -- Assign the output back to the parameters
  ---------------------------------------------------------
  address_name        := l_address_rec.Address_Name;
  address_1           := l_address_rec.Address_Line_1;
  address_2           := l_address_rec.Address_Line_2;
  city                := l_address_rec.City;
  state_province      := NVL(l_address_rec.State, l_address_rec.Province);
  if l_address_rec.Postal_Code_Extended IS NOT NULL then
    postal_code       := l_address_rec.Postal_Code ||'-'||l_address_rec.Postal_Code_Extended;
  else
    postal_code       := l_address_rec.Postal_Code;
  end if;
  country_code        := l_address_rec.Country_Code;
  telephone_number    := l_address_rec.Phone_Number;

  address_status_code := NVL(l_address_rec.Status_Code, ' ')
                      ||' - '||cln$lookup.Get_Address_Status(l_address_rec.Status_Code);
  if l_address_rec.Error_Code IS NULL then
    address_error_code  := l_address_rec.Error_String;
  else
    address_error_code  := NVL(l_address_rec.Error_Code, ' ')
                        ||' - '||l_address_rec.Error_String;
  end if;
  dpv_status_code     := NVL(l_address_rec.DPV_Status, ' ')
                      ||' - '||cln$lookup.Get_DPV_Status(l_address_rec.DPV_Status);
  formatted_address_1 := l_address_rec.Address_Line_1;
  if l_address_rec.Address_Line_2 IS NULL then
    -- move the city state line up to address 2 if address 2 is NULL
    formatted_address_2 := l_address_rec.Address_Line_3;
    formatted_address_3 := NULL;
  else
    formatted_address_2 := l_address_rec.Address_Line_2;
    formatted_address_3 := l_address_rec.Address_Line_3;
  end if;
  formatted_address_4 := l_address_rec.Address_Line_4;
  gender              := NVL(l_address_rec.Gender,' ')
                      ||' - '||cln$lookup.Get_Gender(l_address_rec.Gender);
  telephone_distance  := l_address_rec.Phone_Distance;
  latitude            := l_address_rec.Latitude;
  longitude           := l_address_rec.Longitude;
END Verify;
END CLEAN_Address_Wrapper;
/

set scan on
